package net.bloom.bloomclient.file

import net.bloom.bloomclient.features.mode.Mode
import net.bloom.bloomclient.ui.clickgui.augustus.AugustusClickGUI
import net.bloom.bloomclient.ui.clickgui.augustus.values.AbstractAugustusValueComponent
import net.bloom.bloomclient.value.Value
import net.bloom.bloomclient.value.values.*
import net.minecraft.client.MinecraftInstance

open class Configurable : MinecraftInstance() {
    val augustusComponents = mutableListOf<AbstractAugustusValueComponent<*>>()
    val valuesNotIncludedGroup = mutableListOf<Value<*>>()
    val values = mutableListOf<Value<*>>()

    open val state = true

    fun addValue(value: Value<*>) = values.add(value)

    fun addAugustusComponent(value: Value<*>) {
        val component = AugustusClickGUI.getAugustusComponent(value) ?: return
        augustusComponents.add(component)
    }

    open fun initValues() {
        val groupValues = values.filterIsInstance<GroupValue>().flatMap { it.value.toList() }
        values.removeAll(groupValues.toSet())
        augustusComponents.addAll(values.mapNotNull { AugustusClickGUI.getAugustusComponent(it) })
    }

    protected fun bool(
        name: String,
        value: Boolean,
        description: String = "No description available.",
        applySettingOnThread: Boolean = false,
        canDisplay: () -> Boolean = { true }
    ) = BoolValue(name, description, value, canDisplay, this, applySettingOnThread)

    protected fun int(
        name: String,
        value: Int,
        minRange: Int,
        maxRange: Int,
        suffix: String = "",
        description: String = "No description available.",
        applySettingOnThread: Boolean = false,
        canDisplay: () -> Boolean = { true }
    ) = IntegerValue(name, description, value, minRange, maxRange, suffix, canDisplay, this, applySettingOnThread)

    protected fun float(
        name: String,
        value: Float,
        minRange: Float,
        maxRange: Float,
        suffix: String = "",
        description: String = "No description available.",
        applySettingOnThread: Boolean = false,
        canDisplay: () -> Boolean = { true }
    ) = FloatValue(name, description, value, minRange, maxRange, suffix, canDisplay, this, applySettingOnThread)

    protected fun list(
        name: String,
        value: String,
        values: Array<String>,
        description: String = "No description available.",
        applySettingOnThread: Boolean = false,
        canDisplay: () -> Boolean = { true }
    ) = ListValue(name, description, value, values, canDisplay, this, applySettingOnThread)

    protected fun intRange(
        name: String,
        minValue: Int,
        maxValue: Int,
        minRange: Int,
        maxRange: Int,
        suffix: String = "",
        description: String = "No description available.",
        applySettingOnThread: Boolean = false,
        canDisplay: () -> Boolean = { true }
    ) = IntRangeValue(name, description, minValue, maxValue, minRange, maxRange, suffix, canDisplay, this, applySettingOnThread)

    protected fun floatRange(
        name: String,
        minValue: Float,
        maxValue: Float,
        minRange: Float,
        maxRange: Float,
        suffix: String = "",
        description: String = "No description available.",
        applySettingOnThread: Boolean = false,
        canDisplay: () -> Boolean = { true }
    ) = FloatRangeValue(name, description, minValue, maxValue, minRange, maxRange, suffix, canDisplay, this, applySettingOnThread)

    protected fun key(
        name: String,
        value: Int,
        description: String = "No description available.",
        applySettingOnThread: Boolean = false,
        canDisplay: () -> Boolean = { true }
    ) = KeyValue(name, description, value, canDisplay, this, applySettingOnThread)

    protected fun <T: Mode> mode(
        name: String,
        values: Array<T>,
        description: String = "No description available.",
        applySettingOnThread: Boolean = false,
        canDisplay: () -> Boolean = { true }
    ) = ModeValue(name, description, values, canDisplay, this, applySettingOnThread).also {
        for (mode in it.modes) {
            for (value in mode.values) mode.addAugustusComponent(value)
        }
    }

    protected fun group(
        name: String,
        value: Array<Value<*>>,
        description: String = "No description available.",
        applySettingOnThread: Boolean = false,
        canDisplay: () -> Boolean = { true }
    ) = GroupValue(name, description, value, canDisplay, this, applySettingOnThread)

    protected fun <T: Enum<T>> enum(
        name: String,
        value: T,
        description: String = "No description available.",
        applySettingOnThread: Boolean = false,
        canDisplay: () -> Boolean = { true }
    ) = EnumValue(name, description, value, value.javaClass.enumConstants, canDisplay, this, applySettingOnThread)
}