package net.bloom.bloomclient.features.shared.smoothrotation

import net.bloom.bloomclient.features.mode.Mode
import net.bloom.bloomclient.utils.RandomUtils
import net.bloom.bloomclient.utils.player.RotationUtils
import kotlin.math.cos
import kotlin.math.min
import kotlin.math.sign
import kotlin.random.Random

enum class SmoothTimingMode {
    ANGLE, TIME, NONE
}

abstract class SmoothRotationMode(mode: String) : Mode(mode) {
    // Angle-based calculation (returns signed speed)
    open fun calculateAngle(current: Float, target: Float, min: Float, max: Float): Float {
        return RandomUtils.nextFloat(min, max)
    }

    // Time-based calculation (returns signed rotation amount)
    abstract fun calculate(current: Float, target: Float, minSpeed: Float, maxSpeed: Float, deltaTime: Float): Float

    protected fun calculateSignedRotation(
        current: Float,
        target: Float,
        minSpeed: Float,
        maxSpeed: Float,
        deltaTime: Float,
        easing: (Float) -> Float
    ): Float {
        val diffAngle = RotationUtils.getAngleDifference(current, target)
        val angleFraction = diffAngle / 180f

        val eased = easing(angleFraction)
        val speed = min(maxSpeed, minSpeed + (maxSpeed - minSpeed) * eased)
        val rotation = min(diffAngle, speed * deltaTime)

        return rotation * sign(diffAngle)
    }
}

// Linear implementation
class LinearSmoothRotationMode : SmoothRotationMode("Linear") {
    override fun calculate(current: Float, target: Float, minSpeed: Float, maxSpeed: Float, deltaTime: Float): Float {
        return calculateSignedRotation(current, target, minSpeed, maxSpeed, deltaTime) { it }
    }
}

// Quadratic implementation
class QuadSmoothRotationMode : SmoothRotationMode("Quadratic") {
    override fun calculate(current: Float, target: Float, minSpeed: Float, maxSpeed: Float, deltaTime: Float): Float {
        return calculateSignedRotation(current, target, minSpeed, maxSpeed, deltaTime) { it * it }
    }
}

// Sine implementation
class SineSmoothRotationMode : SmoothRotationMode("Sine") {
    override fun calculate(current: Float, target: Float, minSpeed: Float, maxSpeed: Float, deltaTime: Float): Float {
        return calculateSignedRotation(current, target, minSpeed, maxSpeed, deltaTime) { angleFraction ->
            (-cos(angleFraction * Math.PI).toFloat() * 0.5f + 0.5f)
        }
    }
}

// Quadratic-Sine implementation
class QuadSineSmoothRotationMode : SmoothRotationMode("QuadraticSine") {
    override fun calculate(current: Float, target: Float, minSpeed: Float, maxSpeed: Float, deltaTime: Float): Float {
        return calculateSignedRotation(current, target, minSpeed, maxSpeed, deltaTime) { angleFraction ->
            val cosFraction = (-cos(angleFraction * Math.PI).toFloat() * 0.5f + 0.5f)
            cosFraction * cosFraction
        }
    }
}