package net.bloom.bloomclient.features.shared.rotationspeed

import net.bloom.bloomclient.features.component.components.player.RotationComponent
import net.bloom.bloomclient.features.mode.Mode
import net.bloom.bloomclient.utils.RandomUtils
import net.bloom.bloomclient.utils.extension.safeDiv
import net.bloom.bloomclient.utils.extension.withGCD
import net.bloom.bloomclient.utils.player.RotationUtils
import net.bloom.bloomclient.utils.player.RotationUtils.getAngleDifference
import net.minecraft.util.MathHelper
import net.minecraft.util.Rotation
import kotlin.math.abs
import kotlin.math.hypot
import kotlin.math.min
import kotlin.math.pow


abstract class RotationSpeedMode(mode: String): Mode(mode) {
    abstract fun calculate(targetRotation: Rotation, yawSpeed: Float, pitchSpeed: Float): Rotation
}

class LinearRotationSpeedMode: RotationSpeedMode("Linear") {
    override fun calculate(targetRotation: Rotation, yawSpeed: Float, pitchSpeed: Float): Rotation {
        val yDiff = getAngleDifference(targetRotation.yaw, mc.thePlayer.rotationYaw)
        val pDiff = getAngleDifference(targetRotation.pitch, mc.thePlayer.rotationPitch)

        val rotationDifference = hypot(yDiff, pDiff)

        var (straightLineYaw, straightLinePitch) = run {
            val baseYawSpeed = abs(yDiff safeDiv rotationDifference) * yawSpeed
            val basePitchSpeed = abs(pDiff safeDiv rotationDifference) * pitchSpeed
            baseYawSpeed to basePitchSpeed
        }

        straightLineYaw = yDiff.coerceIn(-straightLineYaw, straightLineYaw)
        straightLinePitch = pDiff.coerceIn(-straightLinePitch, straightLinePitch)

        val targetYaw = mc.thePlayer.rotationYaw + straightLineYaw
        val targetPitch = MathHelper.clamp_float(mc.thePlayer.rotationPitch + straightLinePitch, -90f, 90f)

        return Rotation(targetYaw, targetPitch)
    }
}

/**
 * @author CCBlueX
 */
class HumanizedRotationSpeedMode: RotationSpeedMode("Humanized") {
    private val timing by list("Timing", "OnStart", arrayOf("OnStart", "OnSlowdown", "Always"))

    override fun calculate(targetRotation: Rotation, yawSpeed: Float, pitchSpeed: Float): Rotation {
        val yDiff = getAngleDifference(targetRotation.yaw, mc.thePlayer.rotationYaw)
        val pDiff = getAngleDifference(targetRotation.pitch, mc.thePlayer.rotationPitch)

        val rotationDifference = hypot(yDiff, pDiff)

        var (straightLineYaw, straightLinePitch) = run {
            val baseYawSpeed = abs(yDiff safeDiv rotationDifference) * yawSpeed * RandomUtils.nextFloat(0.9f, 1.1f)
            val basePitchSpeed = abs(pDiff safeDiv rotationDifference) * pitchSpeed * RandomUtils.nextFloat(0.9f, 1.1f)
            baseYawSpeed to basePitchSpeed
        }

        straightLineYaw = yDiff.coerceIn(-straightLineYaw, straightLineYaw)
        straightLinePitch = pDiff.coerceIn(-straightLinePitch, straightLinePitch)

        // Humans usually have some small jitter when moving their mouse from point A to point B.
        // Usually when a rotation axis' difference is prioritized.
        if (rotationDifference > 0F) {
            val yawJitter = RandomUtils.nextFloat(-0.03F, 0.03F) * straightLineYaw
            val pitchJitter = RandomUtils.nextFloat(-0.02F, 0.02F) * straightLinePitch

            straightLineYaw += yawJitter
            straightLinePitch += pitchJitter
        }

        val minYaw = RandomUtils.nextFloat(min(0.1f, RotationUtils.getFixedAngleDelta()), 0.1f).withGCD()
        val minPitch = RandomUtils.nextFloat(min(0.1f, RotationUtils.getFixedAngleDelta()), 0.1f).withGCD()

        applySlowDown(straightLineYaw, minYaw, true) {
            straightLineYaw = it
        }

        applySlowDown(straightLinePitch, minPitch, false) {
            straightLinePitch = it
        }

        val targetYaw = mc.thePlayer.rotationYaw + straightLineYaw
        val targetPitch = MathHelper.clamp_float(mc.thePlayer.rotationPitch + straightLinePitch, -90f, 90f)

        return Rotation(targetYaw, targetPitch)
    }

    private fun applySlowDown(diff: Float, min: Float, yaw: Boolean, action: (Float) -> Unit) {
        if (diff == 0f) {
            action(diff)
            return
        }

        val lastTick1 = if (yaw) {
            getAngleDifference(mc.thePlayer.rotationYaw, mc.thePlayer.prevClientYaw)
        } else {
            getAngleDifference(mc.thePlayer.rotationPitch, mc.thePlayer.prevClientPitch)
        }

        val diffAbs = abs(diff)
        val isSlowingDown = diffAbs <= abs(lastTick1)

        if (diffAbs.withGCD() <= min && (timing == "Always" || timing == "OnSlowDown" && isSlowingDown || timing == "OnStart" && lastTick1 == 0F)) {
            action(0f)
            return
        }

        val range = when (lastTick1) {
            0f -> {
                val inc = 0.2f * (diffAbs / 50f).coerceIn(0f, 1f)

                0.1F + inc..0.5F + inc
            }
            else -> 0.3f..0.7f
        }

        val new = lastTick1 + (diff - lastTick1) * RandomUtils.nextFloat(range.start, range.endInclusive)

        if (abs(new.withGCD()) <= min && isSlowingDown) {
            action(diff)
        } else {
            action(new)
        }
    }

}
//
//class InterpolationRotationSpeedMode: RotationSpeedMode("Interpolation") {
//    private val directionChange by floatRange("DirectionChange", 0f, 0f, 0f, 100f)
//
//    override fun calculate(targetRotation: Rotation, yawSpeed: Float, pitchSpeed: Float): Rotation {
//        val yDiff = getAngleDifference(targetRotation.yaw, mc.thePlayer.rotationYaw)
//        val pDiff = getAngleDifference(targetRotation.pitch, mc.thePlayer.rotationPitch)
//
//        val prevYDiff = getAngleDifference(targetRotation.yaw, mc.thePlayer.prevClientYaw)
//        val prevPDiff = getAngleDifference(targetRotation.pitch, mc.thePlayer.prevClientPitch)
//
//        val directionDiff = hypot(prevYDiff, prevPDiff) * directionChange.random()
//
//        val hSpeed = if (targetRotation == mc.thePlayer.rotation)
//
//    }
//
//}