package net.bloom.bloomclient.features.module

import net.bloom.bloomclient.features.command.Command
import net.bloom.bloomclient.utils.StringUtils
import net.bloom.bloomclient.value.*
import net.bloom.bloomclient.value.range.FloatRange
import net.bloom.bloomclient.value.range.IntegerRange
import net.bloom.bloomclient.value.values.*

class ModuleCommand(val module: Module, val values: List<Value<*>>): Command(module.name.lowercase()) {

	override fun execute(args: Array<String>) {
		val valueNames = values.joinToString("|") { it.name.lowercase() }

		if (args.size < 2) {
			chatSyntax("$command $valueNames <value>")
			return
		}

		val value = values.find { it.name.equals(args[1], true) } ?: run {
			chatSyntax("$command $valueNames <value>")
			return
		}

		if (value is BoolValue) {
            val newValue = !value.get()
            value.set(newValue)

            val state = if (newValue) "on" else "off"

            chat("Value \"${args[1]}\" in module $command was toggled $state.")
            return
        }

        val valueName = args[1].lowercase()

        if (value is IntRangeValue || value is FloatRangeValue) {
            if (args.size < 4) {
                chatSyntax("$command $valueName <min> <max>")
                return
            }
        }

        if (args.size < 3) {
        	when (value) {
        		is IntegerValue, is FloatValue, is TextValue -> chatSyntax("$command $valueName <value>")
        		is ListValue -> {
        			val valuesList = value.values.joinToString("/").lowercase()
        			chatSyntax("$command $valueName <$valuesList>")
        		}
                is ModeValue<*> -> {
                    val valuesList = value.modes.joinToString("/") { it.name }.lowercase()
                    chatSyntax("$command $valueName <$valuesList>")
                }
        	}

            return
        }

        try {
            when (value) {
                is IntegerValue -> value.set(args[2].toInt())
                is FloatValue -> value.set(args[2].toFloat())
                is ListValue -> {
                    if (!value.contains(args[2])) {
                    	val valuesList = value.values.joinToString("/").lowercase()
                        chatSyntax("$command $valueNames <$valuesList>")
                        return
                    }

                    value.set(args[2])
                }
                is TextValue -> value.set(StringUtils.toCompleteString(args, 2))
                is IntRangeValue -> {
                    val newValue = IntegerRange(args[2].toInt(), args[3].toInt()).also { it.correctRange() }
                    value.set(newValue)
                }
                is FloatRangeValue -> {
                    val newValue = FloatRange(args[2].toFloat(), args[3].toFloat()).also { it.correctRange() }
                    value.set(newValue)
                }
                is ModeValue<*> -> if (!value.setValue(args[2])) {
                    val valuesList = value.modes.joinToString("/") { it.name }.lowercase()
                    chatSyntax("$command $valueNames <$valuesList>")
                    return
                }
            }

            chat("§7${module.name} §8${args[1]}§7 was set to §8${value.get()}§7.")
        } catch (e: NumberFormatException) {
            chat("§8${args[2]}§7 cannot be converted to number!")
        }
	}

    override fun tabComplete(args: Array<String>): List<String> {

        if (args.size == 1)
            return values.filter { it.name.startsWith(args.first(), true) }.map { it.name.lowercase() }
        else if (args.size == 2) {
            val value = module.getValue(args[0])

            if (value is ListValue)
                return value.values.filter { it.startsWith(args[1], true) }
        }

        return emptyList()

    }

}