package net.bloom.bloomclient.features.module

import net.bloom.bloomclient.BloomClient
import net.bloom.bloomclient.file.Configurable
import net.bloom.bloomclient.utils.ClientUtils
import net.bloom.bloomclient.utils.animations.Animation
import net.bloom.bloomclient.utils.animations.Easing
import net.bloom.bloomclient.value.Value
import net.bloom.bloomclient.value.values.*
import org.lwjgl.input.Keyboard

open class Module(
    val name: String,
    val displayName: String = name,
    val description: String = "No description for this module.",
    val category: ModuleCategory = ModuleCategory.OTHER,
    var keyBind: Int = Keyboard.CHAR_NONE,
    var showInArray: Boolean = true,
    val onlyEnable: Boolean = false,
    val defaultEnable: Boolean = false
): Configurable() {
    override var state = false
        set(value) {
            if (field == value)
                return

            onToggle(value)

            if (onlyEnable)
                onEnable()
            else if (value) {
                BloomClient.eventManager.register(this)
                registerModeValues()
                onEnable()
            } else {
                BloomClient.eventManager.unregister(this)
                unregisterModeValues()
                onDisable()
            }

            field = !onlyEnable && value
        }

    private val debug = bool("Debug", false)
    val animation = Animation(Easing.EASE_OUT_CUBIC, 500L)

    fun toggle() {
        state = !state
    }

    fun chat(message: String) = ClientUtils.displaySystemMessage(message)

    fun debug(message: String) {
        if (debug.get())
            ClientUtils.displaySystemMessage(message)
    }

    fun getValue(name: String) = values.find {it.name.equals(name, true)}

    open fun onToggle(toggle: Boolean) {}

    private fun registerModeValues() {
        for (value in values) {
            if (value is ModeValue)
                BloomClient.eventManager.register(value.get())
        }
    }

    private fun unregisterModeValues() {
        for (value in values) {
            if (value is ModeValue)
                BloomClient.eventManager.unregister(value.get())
        }
    }

    private fun getValues(valueList: Array<Value<*>>): MutableList<Value<*>> {
        val list = mutableListOf<Value<*>>()
        for (value in valueList) {
            if (value is GroupValue) {
                list.addAll(getValues(value.value))
            } else {
                list.add(value)
                if (value is ModeValue<*>) {
                    val modeValues = value.modes.flatMap { it.values }.toTypedArray()
                    list.addAll(getValues(modeValues))
                }
            }
        }

        return list
    }

    fun generateValuesForCommand() = getValues(values.toTypedArray())

    open fun onEnable() {}

    open fun onDisable() {}

    open fun onInitialize() {}

    open fun onShutdown() {}

    open val tag: String?
        get() = null
}