package net.bloom.bloomclient.features.component

import net.bloom.bloomclient.BloomClient
import net.bloom.bloomclient.utils.ClassUtils
import net.bloom.bloomclient.utils.ClientUtils

class ComponentManager {
	private val componentClassMap = hashMapOf<Class<out Component>, Component>()

	fun registerComponents() {
		ClientUtils.LOGGER.info("[ComponentManager] Loading components...")
		ClassUtils.resolvePackage("${this.javaClass.`package`.name}.components", Component::class.java).forEach(this::registerComponents)
		ClientUtils.LOGGER.info("[ComponentManager] Successfully loaded ${componentClassMap.size} components.")
	}

	fun registerComponents(vararg components: Component) {
		for (component in components)
			registerComponent(component)
	}

	fun registerComponents(vararg componentClasses: Class<out Component>) {
		for (componentClass in componentClasses)
			registerComponent(componentClass)
	}

	fun registerComponent(componentClass: Class<out Component>) {
		try {
			registerComponent(componentClass.getDeclaredConstructor().newInstance())
		} catch (e: IllegalAccessException) {
			registerComponent(ClassUtils.getObjectInstance(componentClass) as Component)
		} catch (e: Exception) {
			ClientUtils.LOGGER.error("Failed to load component: ${componentClass.name}: ${e.message}")
		}
	}

	fun registerComponent(component: Component) {
		componentClassMap[component.javaClass] = component
		BloomClient.eventManager.register(component)
	}

	fun unregisterComponent(component: Component) {
		componentClassMap.remove(component::class.java)
		BloomClient.eventManager.unregister(component)
	}

	operator fun <T: Component> get(clazz: Class<T>) = getComponent(clazz)

	fun <T: Component> getComponent(clazz: Class<T>): T? = componentClassMap[clazz] as T?

}