package net.bloom.bloomclient.features.command

import net.bloom.bloomclient.features.command.commands.BindCommand
import net.bloom.bloomclient.utils.ClassUtils
import net.bloom.bloomclient.utils.ClientUtils
import net.bloom.bloomclient.utils.ClientUtils.LOGGER
import java.util.*

class CommandManager {
	val commands = TreeSet<Command> { command1, command2 -> command1.command.compareTo(command2.command) }
	private val commandClassMap = hashMapOf<Class<*>, Command>()
    var latestCompletions = arrayOf<String>()

	var prefix = "."

	fun registerCommands() {
        LOGGER.info("[ModuleManager] Loading commands...")
        ClassUtils.resolvePackage("${this.javaClass.`package`.name}.commands", Command::class.java)
            .forEach { registerCommand(it.getConstructor().newInstance()) }
        LOGGER.info("[ModuleManager] Successfully loaded ${commands.size} commands.")
    }

    fun executeCommand(input: String) {
    	val args = input.split(" ").toTypedArray()
    	val inputCommand = args.first().removePrefix(prefix)

    	for (command in commands) {
    		if (command.command.equals(inputCommand, true)) {
    			command.execute(args)
    			return
    		}

    		for (alias in command.aliases) {
    			if (alias.equals(inputCommand, true)) {
    				command.execute(args)
    				return
    			}
    		}
    	}

    	ClientUtils.displaySystemMessage("Command not found. Type ${prefix}help to view all commands.")
    }

    fun autoComplete(input: String): Boolean {
        latestCompletions = getCompletions(input)
        return startsPrefix(input) && latestCompletions.isNotEmpty()
    }

    fun getCompletions(input: String): Array<String> {
    	if (input.isEmpty() || !input.startsWith(prefix, true))
    		return arrayOf<String>()

    	val args = input.split(" ").toTypedArray()
    	val inputCommand = args.first().removePrefix(prefix)

    	return if (args.size > 1) {
    		val command = getCommand(inputCommand)
    		val tabCompletions = command?.tabComplete(args.drop(1).toTypedArray())

    		tabCompletions?.toTypedArray() ?: arrayOf<String>()
    	} else {
    		val filteredCommands = commands.filter {
    			it.command.startsWith(inputCommand, true) ||
    			it.aliases.any {alias -> alias.startsWith(inputCommand, true)}
    		}

            val commandsCompletions = mutableSetOf<String>()
            
            for (command in filteredCommands) {
                commandsCompletions += command.command
                commandsCompletions += command.aliases
            }

    		commandsCompletions.map {prefix + it.lowercase()}.toTypedArray()
    	}
    }

    fun startsPrefix(input: String) = input.startsWith(prefix, true)

    fun endsWithCompletions(input: String) = input.endsWith(latestCompletions.last(), true)

    fun isCompletionsEmpty() = latestCompletions.isEmpty()

    fun registerCommands(vararg commands: Command) {
        for (command in commands)
            registerCommand(command)
    }

    fun registerCommands(vararg commandClasses: Class<out Command>) {
        for (commandClass in commandClasses)
            registerCommand(commandClass)
    }

    fun registerCommand(command: Command) {
        commands += command
    }

    fun registerCommand(commandClass: Class<out Command>) {
        try {
            registerCommand(commandClass.getDeclaredConstructor().newInstance())
        } catch (e: IllegalAccessException) {
            registerCommand(ClassUtils.getObjectInstance(commandClass) as Command)
        } catch (e: Exception) {
            ClientUtils.LOGGER.error("Failed to load command: ${commandClass.name}: ${e.message}")
        }
    }

    fun unregisterCommand(command: Command) {
        commands -= command
    }

    fun getCommand(command: String): Command? = commands.find {
    	it.command.equals(command, true) || 
    	it.aliases.any { alias -> alias.equals(command, true) }
    }


}