package net.optifine.shaders.config;

import net.minecraft.src.Config;
import net.optifine.expr.*;
import net.optifine.render.GlAlphaState;
import net.optifine.render.GlBlendState;
import net.optifine.shaders.Program;
import net.optifine.shaders.SMCLog;
import net.optifine.shaders.Shaders;
import net.optifine.shaders.uniform.CustomUniform;
import net.optifine.shaders.uniform.CustomUniforms;
import net.optifine.shaders.uniform.ShaderExpressionResolver;
import net.optifine.shaders.uniform.UniformType;

import java.util.*;

public class ShaderPackParser {
    private static final Map<String, Integer> mapAlphaFuncs = makeMapAlphaFuncs();
    private static final Map<String, Integer> mapBlendFactors = makeMapBlendFactors();

    public static CustomUniforms parseCustomUniforms(Properties props) {
        String s = "uniform";
        String s1 = "variable";
        Map<String, IExpression> map = new HashMap<>();
        List<CustomUniform> list = new ArrayList<>();

        for (Object o : props.keySet()) {
            String s4 = (String) o;
            String[] astring = Config.tokenize(s4, ".");

            if (astring.length == 3) {
                String s5 = astring[0];
                String s6 = astring[1];
                String s7 = astring[2];
                String s8 = props.getProperty(s4).trim();

                if (map.containsKey(s7)) {
                    SMCLog.warning("Expression already defined: " + s7);
                } else if (s5.equals(s) || s5.equals(s1)) {
                    SMCLog.info("Custom " + s5 + ": " + s7);
                    CustomUniform customuniform = parseCustomUniform(s5, s7, s6, s8, map);

                    if (customuniform != null) {
                        map.put(s7, customuniform.getExpression());

                        if (!s5.equals(s1)) {
                            list.add(customuniform);
                        }
                    }
                }
            }
        }

        if (list.isEmpty()) {
            return null;
        } else {
            CustomUniform[] acustomuniform = list.toArray(new CustomUniform[0]);
            return new CustomUniforms(acustomuniform, map);
        }
    }

    private static CustomUniform parseCustomUniform(String kind, String name, String type, String src, Map<String, IExpression> mapExpressions) {
        try {
            UniformType uniformtype = UniformType.parse(type);

            if (uniformtype == null) {
                SMCLog.warning("Unknown " + kind + " type: " + null);
                return null;
            } else {
                ShaderExpressionResolver shaderexpressionresolver = new ShaderExpressionResolver(mapExpressions);
                ExpressionParser expressionparser = new ExpressionParser(shaderexpressionresolver);
                IExpression iexpression = expressionparser.parse(src);
                ExpressionType expressiontype = iexpression.getExpressionType();

                if (!uniformtype.matchesExpressionType(expressiontype)) {
                    SMCLog.warning("Expression type does not match " + kind + " type, expression: " + expressiontype + ", " + kind + ": " + uniformtype + " " + name);
                    return null;
                } else {
                    iexpression = makeExpressionCached(iexpression);
                    return new CustomUniform(name, uniformtype, iexpression);
                }
            }
        } catch (ParseException parseexception) {
            SMCLog.warning(parseexception.getClass().getName() + ": " + parseexception.getMessage());
            return null;
        }
    }

    private static IExpression makeExpressionCached(IExpression expr) {
        return expr instanceof IExpressionFloat ? new ExpressionFloatCached((IExpressionFloat) expr) : (expr instanceof IExpressionFloatArray ? new ExpressionFloatArrayCached((IExpressionFloatArray) expr) : expr);
    }

    public static void parseAlphaStates(Properties props) {
        for (Object o : props.keySet()) {
            String s = (String) o;
            String[] astring = Config.tokenize(s, ".");

            if (astring.length == 2) {
                String s1 = astring[0];
                String s2 = astring[1];

                if (s1.equals("alphaTest")) {
                    Program program = Shaders.getProgram(s2);

                    if (program == null) {
                        SMCLog.severe("Invalid program name: " + s2);
                    } else {
                        String s3 = props.getProperty(s).trim();
                        GlAlphaState glalphastate = parseAlphaState(s3);

                        if (glalphastate != null) {
                            program.setAlphaState(glalphastate);
                        }
                    }
                }
            }
        }
    }

    private static GlAlphaState parseAlphaState(String str) {
        String[] astring = Config.tokenize(str, " ");

        if (astring.length == 1) {
            String s = astring[0];

            if (s.equals("off") || s.equals("false")) {
                return new GlAlphaState(false);
            }
        } else if (astring.length == 2) {
            String s2 = astring[0];
            String s1 = astring[1];
            Integer integer = mapAlphaFuncs.get(s2);
            float f = Config.parseFloat(s1, -1.0F);

            if (integer != null && f >= 0.0F) {
                return new GlAlphaState(true, integer, f);
            }
        }

        SMCLog.severe("Invalid alpha test: " + str);
        return null;
    }

    public static void parseBlendStates(Properties props) {
        for (Object o : props.keySet()) {
            String s = (String) o;
            String[] astring = Config.tokenize(s, ".");

            if (astring.length == 2) {
                String s1 = astring[0];
                String s2 = astring[1];

                if (s1.equals("blend")) {
                    Program program = Shaders.getProgram(s2);

                    if (program == null) {
                        SMCLog.severe("Invalid program name: " + s2);
                    } else {
                        String s3 = props.getProperty(s).trim();
                        GlBlendState glblendstate = parseBlendState(s3);

                        if (glblendstate != null) {
                            program.setBlendState(glblendstate);
                        }
                    }
                }
            }
        }
    }

    private static GlBlendState parseBlendState(String str) {
        String[] astring = Config.tokenize(str, " ");

        if (astring.length == 1) {
            String s = astring[0];

            if (s.equals("off") || s.equals("false")) {
                return new GlBlendState(false);
            }
        } else if (astring.length == 2 || astring.length == 4) {
            String s4 = astring[0];
            String s1 = astring[1];
            String s2 = s4;
            String s3 = s1;

            if (astring.length == 4) {
                s2 = astring[2];
                s3 = astring[3];
            }

            Integer integer = mapBlendFactors.get(s4);
            Integer integer1 = mapBlendFactors.get(s1);
            Integer integer2 = mapBlendFactors.get(s2);
            Integer integer3 = mapBlendFactors.get(s3);

            if (integer != null && integer1 != null && integer2 != null && integer3 != null) {
                return new GlBlendState(true, integer, integer1, integer2, integer3);
            }
        }

        SMCLog.severe("Invalid blend mode: " + str);
        return null;
    }

    public static void parseRenderScales(Properties props) {
        for (Object o : props.keySet()) {
            String s = (String) o;
            String[] astring = Config.tokenize(s, ".");

            if (astring.length == 2) {
                String s1 = astring[0];
                String s2 = astring[1];

                if (s1.equals("scale")) {
                    Program program = Shaders.getProgram(s2);

                    if (program == null) {
                        SMCLog.severe("Invalid program name: " + s2);
                    } else {
                        String s3 = props.getProperty(s).trim();
                        RenderScale renderscale = parseRenderScale(s3);

                        if (renderscale != null) {
                            program.setRenderScale(renderscale);
                        }
                    }
                }
            }
        }
    }

    private static RenderScale parseRenderScale(String str) {
        String[] astring = Config.tokenize(str, " ");
        float f = Config.parseFloat(astring[0], -1.0F);
        float f1 = 0.0F;
        float f2 = 0.0F;

        if (astring.length > 1) {
            if (astring.length != 3) {
                SMCLog.severe("Invalid render scale: " + str);
                return null;
            }

            f1 = Config.parseFloat(astring[1], -1.0F);
            f2 = Config.parseFloat(astring[2], -1.0F);
        }

        if (Config.between(f, 0.0F, 1.0F) && Config.between(f1, 0.0F, 1.0F) && Config.between(f2, 0.0F, 1.0F)) {
            return new RenderScale(f, f1, f2);
        } else {
            SMCLog.severe("Invalid render scale: " + str);
            return null;
        }
    }

    public static void parseBuffersFlip(Properties props) {
        for (Object o : props.keySet()) {
            String s = (String) o;
            String[] astring = Config.tokenize(s, ".");

            if (astring.length == 3) {
                String s1 = astring[0];
                String s2 = astring[1];
                String s3 = astring[2];

                if (s1.equals("flip")) {
                    Program program = Shaders.getProgram(s2);

                    if (program == null) {
                        SMCLog.severe("Invalid program name: " + s2);
                    } else {
                        Boolean[] aboolean = program.getBuffersFlip();
                        int i = Shaders.getBufferIndexFromString(s3);

                        if (i >= 0 && i < aboolean.length) {
                            String s4 = props.getProperty(s).trim();
                            Boolean obool = Config.parseBoolean(s4, null);

                            if (obool == null) {
                                SMCLog.severe("Invalid boolean value: " + s4);
                            } else {
                                aboolean[i] = obool;
                            }
                        } else {
                            SMCLog.severe("Invalid buffer name: " + s3);
                        }
                    }
                }
            }
        }
    }

    private static Map<String, Integer> makeMapAlphaFuncs() {
        return Map.of("NEVER", 512, "LESS", 513, "EQUAL", 514, "LEQUAL", 515, "GREATER", 516, "NOTEQUAL", 517, "GEQUAL", 518, "ALWAYS", 519);
    }

    private static Map<String, Integer> makeMapBlendFactors() {
        return Map.ofEntries(Map.entry("ZERO", 0), Map.entry("ONE", 1), Map.entry("SRC_COLOR", 768), Map.entry("ONE_MINUS_SRC_COLOR", 769), Map.entry("DST_COLOR", 774), Map.entry("ONE_MINUS_DST_COLOR", 775), Map.entry("SRC_ALPHA", 770), Map.entry("ONE_MINUS_SRC_ALPHA", 771), Map.entry("DST_ALPHA", 772), Map.entry("ONE_MINUS_DST_ALPHA", 773), Map.entry("CONSTANT_COLOR", 32769), Map.entry("ONE_MINUS_CONSTANT_COLOR", 32770), Map.entry("CONSTANT_ALPHA", 32771), Map.entry("ONE_MINUS_CONSTANT_ALPHA", 32772), Map.entry("SRC_ALPHA_SATURATE", 776));
    }
}
