package net.minecraft.world.gen.structure;

import com.google.common.collect.Lists;
import net.minecraft.util.BlockPos;
import net.minecraft.util.MathHelper;
import net.minecraft.world.ChunkCoordIntPair;
import net.minecraft.world.World;
import net.minecraft.world.biome.BiomeGenBase;

import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;

public class MapGenStronghold extends MapGenStructure {
    private final List<BiomeGenBase> allowedBiomes;
    private boolean ranBiomeCheck;
    private ChunkCoordIntPair[] structureCoords;
    private double distance;
    private int spread;

    public MapGenStronghold() {
        this.structureCoords = new ChunkCoordIntPair[3];
        this.distance = 32.0D;
        this.spread = 3;
        this.allowedBiomes = Lists.newArrayList();

        for (BiomeGenBase biomegenbase : BiomeGenBase.getBiomeGenArray()) {
            if (biomegenbase != null && biomegenbase.minHeight > 0.0F) {
                this.allowedBiomes.add(biomegenbase);
            }
        }
    }

    public MapGenStronghold(Map<String, String> entryMap) {
        this();

        for (Entry<String, String> entry : entryMap.entrySet()) {
            switch (entry.getKey()) {
                case "distance":
                    this.distance = MathHelper.parseDoubleWithDefaultAndMax(entry.getValue(), this.distance, 1.0D);
                    break;
                case "count":
                    this.structureCoords = new ChunkCoordIntPair[MathHelper.parseIntWithDefaultAndMax(entry.getValue(), this.structureCoords.length, 1)];
                    break;
                case "spread":
                    this.spread = MathHelper.parseIntWithDefaultAndMax(entry.getValue(), this.spread, 1);
                    break;
            }
        }
    }

    public String getStructureName() {
        return "Stronghold";
    }

    protected boolean canSpawnStructureAtCoords(int chunkX, int chunkZ) {
        if (!this.ranBiomeCheck) {
            Random random = new Random();
            random.setSeed(this.worldObj.getSeed());
            double d0 = random.nextDouble() * Math.PI * 2.0D;
            int i = 1;

            for (int j = 0; j < this.structureCoords.length; ++j) {
                double d1 = (1.25D * (double) i + random.nextDouble()) * this.distance * (double) i;
                int k = (int) Math.round(Math.cos(d0) * d1);
                int l = (int) Math.round(Math.sin(d0) * d1);
                BlockPos blockpos = this.worldObj.getWorldChunkManager().findBiomePosition((k << 4) + 8, (l << 4) + 8, 112, this.allowedBiomes, random);

                if (blockpos != null) {
                    k = blockpos.getX() >> 4;
                    l = blockpos.getZ() >> 4;
                }

                this.structureCoords[j] = new ChunkCoordIntPair(k, l);
                d0 += (Math.PI * 2D) * (double) i / (double) this.spread;

                if (j == this.spread) {
                    i += 2 + random.nextInt(5);
                    this.spread += 1 + random.nextInt(2);
                }
            }

            this.ranBiomeCheck = true;
        }

        for (ChunkCoordIntPair chunkcoordintpair : this.structureCoords) {
            if (chunkX == chunkcoordintpair.chunkXPos && chunkZ == chunkcoordintpair.chunkZPos) {
                return true;
            }
        }

        return false;
    }

    protected List<BlockPos> getCoordList() {
        List<BlockPos> list = Lists.newArrayList();

        for (ChunkCoordIntPair chunkcoordintpair : this.structureCoords) {
            if (chunkcoordintpair != null) {
                list.add(chunkcoordintpair.getCenterBlock(64));
            }
        }

        return list;
    }

    protected StructureStart getStructureStart(int chunkX, int chunkZ) {
        MapGenStronghold.Start mapgenstronghold$start;

        for (mapgenstronghold$start = new MapGenStronghold.Start(this.worldObj, this.rand, chunkX, chunkZ); mapgenstronghold$start.getComponents().isEmpty() || ((StructureStrongholdPieces.Stairs2) mapgenstronghold$start.getComponents().get(0)).strongholdPortalRoom == null; mapgenstronghold$start = new MapGenStronghold.Start(this.worldObj, this.rand, chunkX, chunkZ)) {
        }

        return mapgenstronghold$start;
    }

    public static class Start extends StructureStart {
        public Start() {
        }

        public Start(World worldIn, Random random, int chunkX, int chunkZ) {
            super(chunkX, chunkZ);
            StructureStrongholdPieces.prepareStructurePieces();
            StructureStrongholdPieces.Stairs2 structurestrongholdpieces$stairs2 = new StructureStrongholdPieces.Stairs2(0, random, (chunkX << 4) + 2, (chunkZ << 4) + 2);
            this.components.add(structurestrongholdpieces$stairs2);
            structurestrongholdpieces$stairs2.buildComponent(structurestrongholdpieces$stairs2, this.components, random);
            List<StructureComponent> list = structurestrongholdpieces$stairs2.pendingChildren;

            while (!list.isEmpty()) {
                int i = random.nextInt(list.size());
                StructureComponent structurecomponent = list.remove(i);
                structurecomponent.buildComponent(structurestrongholdpieces$stairs2, this.components, random);
            }

            this.updateBoundingBox();
            this.markAvailableHeight(worldIn, random, 10);
        }
    }
}
