package net.minecraft.util;

import net.minecraft.client.option.options.videosettings.PerformanceOption;
import net.optifine.util.MathUtils;

import java.util.Random;
import java.util.UUID;

public class MathHelper {
    public static final float PI = MathUtils.roundToFloat(Math.PI);
    public static final float PId2 = MathUtils.roundToFloat((Math.PI / 2D));
    public static final float deg2Rad = MathUtils.roundToFloat(0.017453292519943295D);
    private static final float radToIndex = MathUtils.roundToFloat(651.8986469044033D);
    private static final float[] SIN_TABLE_FAST = new float[4096];
    private static final float[] SIN_TABLE = new float[65536];
    private static final int[] multiplyDeBruijnBitPosition;
    private static final double FRAC_BIAS;
    private static final double[] ASINE_TAB;
    private static final double[] COS_TAB;

    static {
        for (int i = 0; i < 65536; ++i) {
            SIN_TABLE[i] = (float) Math.sin((double) i * Math.PI * 2.0D / 65536.0D);
        }

        for (int j = 0; j < SIN_TABLE_FAST.length; ++j) {
            SIN_TABLE_FAST[j] = MathUtils.roundToFloat(Math.sin((double) j * Math.PI * 2.0D / 4096.0D));
        }

        multiplyDeBruijnBitPosition = new int[]{0, 1, 28, 2, 29, 14, 24, 3, 30, 22, 20, 15, 25, 17, 4, 8, 31, 27, 13, 23, 21, 19, 16, 7, 26, 12, 18, 6, 11, 5, 10, 9};
        FRAC_BIAS = Double.longBitsToDouble(4805340802404319232L);
        ASINE_TAB = new double[257];
        COS_TAB = new double[257];

        for (int k = 0; k < 257; ++k) {
            double d0 = (double) k / 256.0D;
            double d1 = Math.asin(d0);
            COS_TAB[k] = Math.cos(d1);
            ASINE_TAB[k] = d1;
        }
    }

    public static float sin(float value) {
        return PerformanceOption.isFastMath() ? SIN_TABLE_FAST[(int) (value * radToIndex) & 4095] : SIN_TABLE[(int) (value * 10430.378F) & 65535];
    }

    public static float cos(float value) {
        return PerformanceOption.isFastMath() ? SIN_TABLE_FAST[(int) (value * radToIndex + 1024.0F) & 4095] : SIN_TABLE[(int) (value * 10430.378F + 16384.0F) & 65535];
    }

    public static float sqrt_float(float value) {
        return (float) Math.sqrt(value);
    }

    public static float sqrt_double(double value) {
        return (float) Math.sqrt(value);
    }

    public static int floor_float(float value) {
        int i = (int) value;
        return value < (float) i ? i - 1 : i;
    }

    public static int truncateDoubleToInt(double value) {
        return (int) (value + 1024.0D) - 1024;
    }

    public static int floor_double(double value) {
        int i = (int) value;
        return value < (double) i ? i - 1 : i;
    }

    public static long floor_double_long(double value) {
        long i = (long) value;
        return value < (double) i ? i - 1L : i;
    }

    public static int absFloor(double value) {
        return (int) (value >= 0.0D ? value : -value + 1.0D);
    }

    public static float abs(float value) {
        return value >= 0.0F ? value : -value;
    }

    public static int abs_int(int value) {
        return value >= 0 ? value : -value;
    }

    public static int ceiling_float_int(float value) {
        int i = (int) value;
        return value > (float) i ? i + 1 : i;
    }

    public static int ceiling_double_int(double value) {
        int i = (int) value;
        return value > (double) i ? i + 1 : i;
    }

    public static int clamp_int(int num, int min, int max) {
        return num < min ? min : (Math.min(num, max));
    }

    public static float clamp_float(float num, float min, float max) {
        return num < min ? min : (Math.min(num, max));
    }

    public static double clamp_double(double num, double min, double max) {
        return num < min ? min : (Math.min(num, max));
    }

    public static double denormalizeClamp(double lowerBnd, double upperBnd, double slide) {
        return slide < 0.0D ? lowerBnd : (slide > 1.0D ? upperBnd : lowerBnd + (upperBnd - lowerBnd) * slide);
    }

    public static double abs_max(double x, double y) {
        if (x < 0.0D) {
            x = -x;
        }

        if (y < 0.0D) {
            y = -y;
        }

        return Math.max(x, y);
    }

    public static int bucketInt(int x, int y) {
        return x < 0 ? -((-x - 1) / y) - 1 : x / y;
    }

    public static int getRandomIntegerInRange(Random random, int minimum, int maximum) {
        return minimum >= maximum ? minimum : random.nextInt(maximum - minimum + 1) + minimum;
    }

    public static float randomFloatClamp(Random random, float minimum, float maximum) {
        return minimum >= maximum ? minimum : random.nextFloat() * (maximum - minimum) + minimum;
    }

    public static double getRandomDoubleInRange(Random random, double minimum, double maximum) {
        return minimum >= maximum ? minimum : random.nextDouble() * (maximum - minimum) + minimum;
    }

    public static double average(long[] values) {
        long i = 0L;

        for (long j : values) {
            i += j;
        }

        return (double) i / (double) values.length;
    }

    public static boolean epsilonEquals(float x, float y) {
        return abs(y - x) < 1.0E-5F;
    }

    public static int normalizeAngle(int x, int y) {
        return (x % y + y) % y;
    }

    public static float wrapAngleTo180(float value) {
        value = value % 360.0F;

        if (value >= 180.0F) {
            value -= 360.0F;
        }

        if (value < -180.0F) {
            value += 360.0F;
        }

        return value;
    }

    public static double wrapAngleTo180(double value) {
        value = value % 360.0D;

        if (value >= 180.0D) {
            value -= 360.0D;
        }

        if (value < -180.0D) {
            value += 360.0D;
        }

        return value;
    }

    public static int parseIntWithDefault(String value, int defaultValue) {
        try {
            return Integer.parseInt(value);
        } catch (Throwable var3) {
            return defaultValue;
        }
    }

    public static int parseIntWithDefaultAndMax(String value, int defaultValue, int max) {
        return Math.max(max, parseIntWithDefault(value, defaultValue));
    }

    public static double parseDoubleWithDefault(String value, double defaultValue) {
        try {
            return Double.parseDouble(value);
        } catch (Throwable var4) {
            return defaultValue;
        }
    }

    public static double parseDoubleWithDefaultAndMax(String value, double defaultValue, double max) {
        return Math.max(max, parseDoubleWithDefault(value, defaultValue));
    }

    public static int roundUpToPowerOfTwo(int value) {
        int i = value - 1;
        i = i | i >> 1;
        i = i | i >> 2;
        i = i | i >> 4;
        i = i | i >> 8;
        i = i | i >> 16;
        return i + 1;
    }

    private static boolean isPowerOfTwo(int value) {
        return value != 0 && (value & value - 1) == 0;
    }

    private static int calculateLogBaseTwoDeBruijn(int value) {
        value = isPowerOfTwo(value) ? value : roundUpToPowerOfTwo(value);
        return multiplyDeBruijnBitPosition[(int) ((long) value * 125613361L >> 27) & 31];
    }

    public static int calculateLogBaseTwo(int value) {
        return calculateLogBaseTwoDeBruijn(value) - (isPowerOfTwo(value) ? 0 : 1);
    }

    public static int roundUp(int number, int interval) {
        if (interval == 0) {
            return 0;
        } else if (number == 0) {
            return interval;
        } else {
            if (number < 0) {
                interval *= -1;
            }

            int i = number % interval;
            return i == 0 ? number : number + interval - i;
        }
    }

    public static int rgb(float rIn, float gIn, float bIn) {
        return rgb(floor_float(rIn * 255.0F), floor_float(gIn * 255.0F), floor_float(bIn * 255.0F));
    }

    public static int rgb(int rIn, int gIn, int bIn) {
        int i = (rIn << 8) + gIn;
        i = (i << 8) + bIn;
        return i;
    }

    public static int multiplyColor(int color1, int color2) {
        int i = (color1 & 16711680) >> 16;
        int j = (color2 & 16711680) >> 16;
        int k = (color1 & 65280) >> 8;
        int l = (color2 & 65280) >> 8;
        int i1 = (color1 & 255);
        int j1 = (color2 & 255);
        int k1 = (int) ((float) i * (float) j / 255.0F);
        int l1 = (int) ((float) k * (float) l / 255.0F);
        int i2 = (int) ((float) i1 * (float) j1 / 255.0F);
        return color1 & -16777216 | k1 << 16 | l1 << 8 | i2;
    }

    public static double frac(double number) {
        return number - Math.floor(number);
    }

    public static long getPositionRandom(Vec3i pos) {
        return getCoordinateRandom(pos.getX(), pos.getY(), pos.getZ());
    }

    public static long getCoordinateRandom(int x, int y, int z) {
        long i = (x * 3129871L) ^ (long) z * 116129781L ^ (long) y;
        i = i * i * 42317861L + i * 11L;
        return i;
    }

    public static UUID getRandomUuid(Random rand) {
        long i = rand.nextLong() & -61441L | 16384L;
        long j = rand.nextLong() & 4611686018427387903L | Long.MIN_VALUE;
        return new UUID(i, j);
    }

    public static double pct(double current, double min, double max) {
        return (current - min) / (max - min);
    }

    public static double atan2(double y, double x) {
        double d0 = x * x + y * y;

        if (Double.isNaN(d0)) {
            return Double.NaN;
        } else {
            boolean flag = y < 0.0D;

            if (flag) {
                y = -y;
            }

            boolean flag1 = x < 0.0D;

            if (flag1) {
                x = -x;
            }

            boolean flag2 = y > x;

            if (flag2) {
                double d1 = x;
                x = y;
                y = d1;
            }

            double d9 = fastInvSqrt(d0);
            x = x * d9;
            y = y * d9;
            double d2 = FRAC_BIAS + y;
            int i = (int) Double.doubleToRawLongBits(d2);
            double d3 = ASINE_TAB[i];
            double d4 = COS_TAB[i];
            double d5 = d2 - FRAC_BIAS;
            double d6 = y * d4 - x * d5;
            double d7 = (6.0D + d6 * d6) * d6 * 0.16666666666666666D;
            double d8 = d3 + d7;

            if (flag2) {
                d8 = (Math.PI / 2D) - d8;
            }

            if (flag1) {
                d8 = Math.PI - d8;
            }

            if (flag) {
                d8 = -d8;
            }

            return d8;
        }
    }

    public static double fastInvSqrt(double number) {
        double d0 = 0.5D * number;
        long i = Double.doubleToRawLongBits(number);
        i = 6910469410427058090L - (i >> 1);
        number = Double.longBitsToDouble(i);
        number = number * (1.5D - d0 * number * number);
        return number;
    }

    public static int hsvToRGB(float hue, float saturation, float value) {
        int i = (int) (hue * 6.0F) % 6;
        float f = hue * 6.0F - (float) i;
        float f1 = value * (1.0F - saturation);
        float f2 = value * (1.0F - f * saturation);
        float f3 = value * (1.0F - (1.0F - f) * saturation);
        float f4;
        float f5;
        float f6;

        switch (i) {
            case 0 -> {
                f4 = value;
                f5 = f3;
                f6 = f1;
            }
            case 1 -> {
                f4 = f2;
                f5 = value;
                f6 = f1;
            }
            case 2 -> {
                f4 = f1;
                f5 = value;
                f6 = f3;
            }
            case 3 -> {
                f4 = f1;
                f5 = f2;
                f6 = value;
            }
            case 4 -> {
                f4 = f3;
                f5 = f1;
                f6 = value;
            }
            case 5 -> {
                f4 = value;
                f5 = f1;
                f6 = f2;
            }
            default -> throw new RuntimeException("Something went wrong when converting from HSV to RGB. Input was " + hue + ", " + saturation + ", " + value);
        }

        int j = clamp_int((int) (f4 * 255.0F), 0, 255);
        int k = clamp_int((int) (f5 * 255.0F), 0, 255);
        int l = clamp_int((int) (f6 * 255.0F), 0, 255);
        return j << 16 | k << 8 | l;
    }

    // Start of BloomClient

    public static Vec3 getNearestPointBB(Vec3 eye, AxisAlignedBB box) {
        double[] origin = { eye.xCoord, eye.yCoord, eye.zCoord };
        double[] destMins = { box.minX, box.minY, box.minZ };
        double[] destMaxs = { box.maxX, box.maxY, box.maxZ };

        for (int i = 0; i < 3; i++) {
            if (origin[i] > destMaxs[i]) {
                origin[i] = destMaxs[i];
            } else if (origin[i] < destMins[i]){
                origin[i] = destMins[i];
            }
        }

        return new Vec3(origin[0], origin[1], origin[2]);
    }

    // End of BloomClient
}
