package net.minecraft.util;

import com.google.common.collect.Lists;
import net.minecraft.block.Block;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Blocks;
import net.minecraft.item.ItemStack;

import java.util.List;

public class CombatTracker {
    private final List<CombatEntry> combatEntries = Lists.newArrayList();
    private final EntityLivingBase fighter;
    private int lastDamageTime;
    private int combatStartTime;
    private int combatEndTime;
    private boolean inCombat;
    private boolean takingDamage;
    private String fallSuffix;

    public CombatTracker(EntityLivingBase fighterIn) {
        this.fighter = fighterIn;
    }

    public void calculateFallSuffix() {
        this.resetFallSuffix();

        if (this.fighter.isOnLadder()) {
            Block block = this.fighter.worldObj.getBlockState(new BlockPos(this.fighter.posX, this.fighter.getEntityBoundingBox().minY, this.fighter.posZ)).getBlock();

            if (block == Blocks.ladder) {
                this.fallSuffix = "ladder";
            } else if (block == Blocks.vine) {
                this.fallSuffix = "vines";
            }
        } else if (this.fighter.isInWater()) {
            this.fallSuffix = "water";
        }
    }

    public void trackDamage(DamageSource damageSrc, float healthIn, float damageAmount) {
        this.reset();
        this.calculateFallSuffix();
        CombatEntry combatentry = new CombatEntry(damageSrc, this.fighter.ticksExisted, healthIn, damageAmount, this.fallSuffix, this.fighter.fallDistance);
        this.combatEntries.add(combatentry);
        this.lastDamageTime = this.fighter.ticksExisted;
        this.takingDamage = true;

        if (combatentry.isLivingDamageSrc() && !this.inCombat && this.fighter.isEntityAlive()) {
            this.inCombat = true;
            this.combatStartTime = this.fighter.ticksExisted;
            this.combatEndTime = this.combatStartTime;
            this.fighter.sendEnterCombat();
        }
    }

    public IChatComponent getDeathMessage() {
        if (this.combatEntries.isEmpty()) {
            return new ChatComponentTranslation("death.attack.generic", this.fighter.getDisplayName());
        } else {
            CombatEntry combatentry = this.getBestCombatEntry();
            CombatEntry combatentry1 = this.combatEntries.get(this.combatEntries.size() - 1);
            IChatComponent ichatcomponent1 = combatentry1.getDamageSrcDisplayName();
            Entity entity = combatentry1.getDamageSrc().getEntity();
            IChatComponent ichatcomponent;

            if (combatentry != null && combatentry1.getDamageSrc() == DamageSource.fall) {
                IChatComponent ichatcomponent2 = combatentry.getDamageSrcDisplayName();

                if (combatentry.getDamageSrc() != DamageSource.fall && combatentry.getDamageSrc() != DamageSource.outOfWorld) {
                    if (ichatcomponent2 != null && (!ichatcomponent2.equals(ichatcomponent1))) {
                        Entity entity1 = combatentry.getDamageSrc().getEntity();
                        ItemStack itemstack1 = entity1 instanceof EntityLivingBase ? ((EntityLivingBase) entity1).getHeldItem() : null;

                        if (itemstack1 != null && itemstack1.hasDisplayName()) {
                            ichatcomponent = new ChatComponentTranslation("death.fell.assist.item", this.fighter.getDisplayName(), ichatcomponent2, itemstack1.getChatComponent());
                        } else {
                            ichatcomponent = new ChatComponentTranslation("death.fell.assist", this.fighter.getDisplayName(), ichatcomponent2);
                        }
                    } else if (ichatcomponent1 != null) {
                        ItemStack itemstack = entity instanceof EntityLivingBase ? ((EntityLivingBase) entity).getHeldItem() : null;

                        if (itemstack != null && itemstack.hasDisplayName()) {
                            ichatcomponent = new ChatComponentTranslation("death.fell.finish.item", this.fighter.getDisplayName(), ichatcomponent1, itemstack.getChatComponent());
                        } else {
                            ichatcomponent = new ChatComponentTranslation("death.fell.finish", this.fighter.getDisplayName(), ichatcomponent1);
                        }
                    } else {
                        ichatcomponent = new ChatComponentTranslation("death.fell.killer", this.fighter.getDisplayName());
                    }
                } else {
                    ichatcomponent = new ChatComponentTranslation("death.fell.accident." + this.getFallSuffix(combatentry), this.fighter.getDisplayName());
                }
            } else {
                ichatcomponent = combatentry1.getDamageSrc().getDeathMessage(this.fighter);
            }

            return ichatcomponent;
        }
    }

    public EntityLivingBase getBestAttacker() {
        EntityLivingBase entitylivingbase = null;
        EntityPlayer entityplayer = null;
        float f = 0.0F;
        float f1 = 0.0F;

        for (CombatEntry combatentry : this.combatEntries) {
            if (combatentry.getDamageSrc().getEntity() instanceof EntityPlayer && (entityplayer == null || combatentry.getDamage() > f1)) {
                f1 = combatentry.getDamage();
                entityplayer = (EntityPlayer) combatentry.getDamageSrc().getEntity();
            }

            if (combatentry.getDamageSrc().getEntity() instanceof EntityLivingBase && (entitylivingbase == null || combatentry.getDamage() > f)) {
                f = combatentry.getDamage();
                entitylivingbase = (EntityLivingBase) combatentry.getDamageSrc().getEntity();
            }
        }

        if (entityplayer != null && f1 >= f / 3.0F) {
            return entityplayer;
        } else {
            return entitylivingbase;
        }
    }

    private CombatEntry getBestCombatEntry() {
        CombatEntry combatentry = null;
        CombatEntry combatentry1 = null;
        int i = 0;
        float f = 0.0F;

        for (int j = 0; j < this.combatEntries.size(); ++j) {
            CombatEntry combatentry2 = this.combatEntries.get(j);
            CombatEntry combatentry3 = j > 0 ? this.combatEntries.get(j - 1) : null;

            if ((combatentry2.getDamageSrc() == DamageSource.fall || combatentry2.getDamageSrc() == DamageSource.outOfWorld) && combatentry2.getDamageAmount() > 0.0F && (combatentry == null || combatentry2.getDamageAmount() > f)) {
                if (j > 0) {
                    combatentry = combatentry3;
                } else {
                    combatentry = combatentry2;
                }

                f = combatentry2.getDamageAmount();
            }

            if (combatentry2.getFallSuffix() != null && (combatentry1 == null || combatentry2.getDamage() > (float) i)) {
                combatentry1 = combatentry2;
            }
        }

        if (f > 5.0F && combatentry != null) {
            return combatentry;
        } else {
            return null;
        }
    }

    private String getFallSuffix(CombatEntry entry) {
        return entry.getFallSuffix() == null ? "generic" : entry.getFallSuffix();
    }

    public int getCombatDuration() {
        return this.inCombat ? this.fighter.ticksExisted - this.combatStartTime : this.combatEndTime - this.combatStartTime;
    }

    private void resetFallSuffix() {
        this.fallSuffix = null;
    }

    public void reset() {
        int i = this.inCombat ? 300 : 100;

        if (this.takingDamage && (!this.fighter.isEntityAlive() || this.fighter.ticksExisted - this.lastDamageTime > i)) {
            boolean flag = this.inCombat;
            this.takingDamage = false;
            this.inCombat = false;
            this.combatEndTime = this.fighter.ticksExisted;

            if (flag) {
                this.fighter.sendEndCombat();
            }

            this.combatEntries.clear();
        }
    }

    public EntityLivingBase getFighter() {
        return this.fighter;
    }
}
