package net.minecraft.stats;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonParser;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.network.play.server.S37PacketStatistics;
import net.minecraft.server.MinecraftServer;
import net.minecraft.util.ChatComponentTranslation;
import net.minecraft.util.IJsonSerializable;
import net.minecraft.util.TupleIntJsonSerializable;
import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.nio.charset.Charset;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

public class StatisticsFile extends StatFileWriter {
    private static final Logger logger = LogManager.getLogger();
    private final MinecraftServer mcServer;
    private final File statsFile;
    private final Set<StatBase> dirty = Sets.newHashSet();
    private int lastStatRequest = -300;
    private boolean hasUnsentAchievement = false;

    public StatisticsFile(MinecraftServer serverIn, File statsFileIn) {
        this.mcServer = serverIn;
        this.statsFile = statsFileIn;
    }

    public static String dumpJson(Map<StatBase, TupleIntJsonSerializable> statMap) {
        JsonObject jsonobject = new JsonObject();

        for (Entry<StatBase, TupleIntJsonSerializable> entry : statMap.entrySet()) {
            if (entry.getValue().getJsonSerializableValue() != null) {
                JsonObject jsonobject1 = new JsonObject();
                jsonobject1.addProperty("value", entry.getValue().getint());

                try {
                    jsonobject1.add("progress", entry.getValue().getJsonSerializableValue().getSerializableElement());
                } catch (Throwable throwable) {
                    logger.warn("Couldn't save statistic {}: error serializing progress", entry.getKey().getStatName(), throwable);
                }

                jsonobject.add(entry.getKey().statId, jsonobject1);
            } else {
                jsonobject.addProperty(entry.getKey().statId, entry.getValue().getint());
            }
        }

        return jsonobject.toString();
    }

    public void readStatFile() {
        if (this.statsFile.isFile()) {
            try {
                this.statsData.clear();
                this.statsData.putAll(this.parseJson(FileUtils.readFileToString(this.statsFile, Charset.defaultCharset())));
            } catch (IOException ioexception) {
                logger.error("Couldn't read statistics file {}", this.statsFile, ioexception);
            } catch (JsonParseException jsonparseexception) {
                logger.error("Couldn't parse statistics file {}", this.statsFile, jsonparseexception);
            }
        }
    }

    public void saveStatFile() {
        try {
            FileUtils.writeStringToFile(this.statsFile, dumpJson(this.statsData), Charset.defaultCharset());
        } catch (IOException ioexception) {
            logger.error("Couldn't save stats", ioexception);
        }
    }

    public void unlockAchievement(EntityPlayer playerIn, StatBase statIn, int achievementValue) {
        int i = statIn.isAchievement() ? this.readStat(statIn) : 0;
        super.unlockAchievement(playerIn, statIn, achievementValue);
        this.dirty.add(statIn);

        if (statIn.isAchievement() && i == 0 && achievementValue > 0) {
            this.hasUnsentAchievement = true;

            if (this.mcServer.isAnnouncingPlayerAchievements()) {
                this.mcServer.getConfigurationManager().sendChatMsg(new ChatComponentTranslation("chat.type.achievement", playerIn.getDisplayName(), statIn.createChatComponent()));
            }
        }

        if (statIn.isAchievement() && i > 0 && achievementValue == 0) {
            this.hasUnsentAchievement = true;

            if (this.mcServer.isAnnouncingPlayerAchievements()) {
                this.mcServer.getConfigurationManager().sendChatMsg(new ChatComponentTranslation("chat.type.achievement.taken", playerIn.getDisplayName(), statIn.createChatComponent()));
            }
        }
    }

    public Set<StatBase> getDirty() {
        Set<StatBase> set = Sets.newHashSet(this.dirty);
        this.dirty.clear();
        this.hasUnsentAchievement = false;
        return set;
    }

    public Map<StatBase, TupleIntJsonSerializable> parseJson(String jsonString) {
        JsonElement jsonelement = JsonParser.parseString(jsonString);

        if (!jsonelement.isJsonObject()) {
            return Maps.newHashMap();
        } else {
            JsonObject jsonobject = jsonelement.getAsJsonObject();
            Map<StatBase, TupleIntJsonSerializable> map = Maps.newHashMap();

            for (Entry<String, JsonElement> entry : jsonobject.entrySet()) {
                StatBase statbase = StatList.getOneShotStat(entry.getKey());

                if (statbase != null) {
                    TupleIntJsonSerializable tupleintjsonserializable = new TupleIntJsonSerializable();

                    if (entry.getValue().isJsonPrimitive() && entry.getValue().getAsJsonPrimitive().isNumber()) {
                        tupleintjsonserializable.setint(entry.getValue().getAsInt());
                    } else if (entry.getValue().isJsonObject()) {
                        JsonObject jsonobject1 = entry.getValue().getAsJsonObject();

                        if (jsonobject1.has("value") && jsonobject1.get("value").isJsonPrimitive() && jsonobject1.get("value").getAsJsonPrimitive().isNumber()) {
                            tupleintjsonserializable.setint(jsonobject1.getAsJsonPrimitive("value").getAsInt());
                        }

                        if (jsonobject1.has("progress") && statbase.getSerializableClazz() != null) {
                            try {
                                Constructor<? extends IJsonSerializable> constructor = statbase.getSerializableClazz().getConstructor();
                                IJsonSerializable ijsonserializable = constructor.newInstance();
                                ijsonserializable.fromJson(jsonobject1.get("progress"));
                                tupleintjsonserializable.setJsonSerializableValue(ijsonserializable);
                            } catch (Throwable throwable) {
                                logger.warn("Invalid statistic progress in {}", this.statsFile, throwable);
                            }
                        }
                    }

                    map.put(statbase, tupleintjsonserializable);
                } else {
                    logger.warn("Invalid statistic in {}: Don't know what {} is", this.statsFile, entry.getKey());
                }
            }

            return map;
        }
    }

    public void markAllDirty() {
        this.dirty.addAll(this.statsData.keySet());
    }

    public void sendStats(EntityPlayerMP player) {
        int i = this.mcServer.getTickCounter();
        Map<StatBase, Integer> map = Maps.newHashMap();

        if (this.hasUnsentAchievement || i - this.lastStatRequest > 300) {
            this.lastStatRequest = i;

            for (StatBase statbase : this.getDirty()) {
                map.put(statbase, this.readStat(statbase));
            }
        }

        player.playerNetServerHandler.sendPacket(new S37PacketStatistics(map));
    }

    public void sendAchievements(EntityPlayerMP player) {
        Map<StatBase, Integer> map = Maps.newHashMap();

        for (Achievement achievement : AchievementList.achievementList) {
            if (this.hasAchievementUnlocked(achievement)) {
                map.put(achievement, this.readStat(achievement));
                this.dirty.remove(achievement);
            }
        }

        player.playerNetServerHandler.sendPacket(new S37PacketStatistics(map));
    }

    public boolean hasUnsentAchievement() {
        return this.hasUnsentAchievement;
    }
}
