package net.minecraft.src;

import net.minecraft.client.LoadingScreenRenderer;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.ScaledResolution;
import net.minecraft.client.option.options.general.DebuggingOption;
import net.minecraft.client.option.options.general.GeneralOption;
import net.minecraft.client.option.options.general.OtherOption;
import net.minecraft.client.option.options.videosettings.PerformanceOption;
import net.minecraft.client.option.options.videosettings.QualityOption;
import net.minecraft.client.renderer.GLAllocation;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.OpenGlHelper;
import net.minecraft.client.renderer.RenderGlobal;
import net.minecraft.client.renderer.texture.DynamicTexture;
import net.minecraft.client.renderer.texture.TextureManager;
import net.minecraft.client.renderer.texture.TextureMap;
import net.minecraft.client.resources.*;
import net.minecraft.client.resources.model.ModelManager;
import net.minecraft.util.*;
import net.optifine.DynamicLights;
import net.optifine.GlErrors;
import net.optifine.config.GlVersion;
import net.optifine.shaders.Shaders;
import net.optifine.util.DisplayModeComparator;
import net.optifine.util.PropertiesOrdered;
import net.optifine.util.TextureUtils;
import net.optifine.util.TimedEvent;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.lwjgl.Sys;
import org.lwjgl.opengl.DisplayMode;
import org.lwjgl.opengl.*;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.*;
import java.lang.reflect.Array;
import java.nio.ByteBuffer;
import java.nio.IntBuffer;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Config {
    public static final boolean logDetail = System.getProperty("log.detail", "false").equals("true");
    private static final Minecraft minecraft = Minecraft.getMinecraft();
    private static final Logger LOGGER = LogManager.getLogger();
    public static String openGlVersion = null;
    public static String openGlRenderer = null;
    public static String openGlVendor = null;
    public static String[] openGlExtensions = null;
    public static GlVersion glVersion = null;
    public static GlVersion glslVersion = null;
    public static int minecraftVersionInt = -1;
    public static boolean fancyFogAvailable = false;
    public static boolean occlusionAvailable = false;
    public static boolean zoomMode = false;
    public static boolean zoomSmoothCamera = false;
    public static boolean waterOpacityChanged = false;
    public static float renderPartialTicks;
    private static String build = null;
    private static boolean notify64BitJava = false;
    private static boolean initialized = false;
    private static Thread minecraftThread = null;
    public static DisplayMode desktopDisplayMode = null;
    private static DisplayMode[] displayModes = null;
    private static int availableProcessors = 0;
    public static int texturePackClouds = 0;
    private static boolean fullscreenModeChecked = false;
    private static boolean desktopModeChecked = false;
    private static DefaultResourcePack defaultResourcePackLazy = null;
    private static String mcDebugLast = null;
    private static int fpsMinLast = 0;

    public static String getVersion() {
        return "OptiFine_1.8.9_HD_U_M6_pre2";
    }

    public static String getVersionDebug() {
        StringBuilder stringbuffer = new StringBuilder(32);

        if (QualityOption.isDynamicLights()) {
            stringbuffer.append("DL: ");
            stringbuffer.append(DynamicLights.getCount());
            stringbuffer.append(", ");
        }

        stringbuffer.append("OptiFine_1.8.9_HD_U_M6_pre2");
        String s = Shaders.getShaderPackName();

        if (s != null) {
            stringbuffer.append(", ");
            stringbuffer.append(s);
        }

        return stringbuffer.toString();
    }

    public static void initDisplay() {
        checkInitialized();
        checkDisplayMode();
        minecraftThread = Thread.currentThread();
        updateThreadPriorities();
        Shaders.startup();
    }

    public static void checkInitialized() {
        if (!initialized) {
            if (Display.isCreated()) {
                initialized = true;
                checkOpenGlCaps();
            }
        }
    }

    private static void checkOpenGlCaps() {
        log("");
        log(getVersion());
        log("Build: " + getBuild());
        log("OS: " + System.getProperty("os.name") + " (" + System.getProperty("os.arch") + ") version " + System.getProperty("os.version"));
        log("Java: " + System.getProperty("java.version") + ", " + System.getProperty("java.vendor"));
        log("VM: " + System.getProperty("java.vm.name") + " (" + System.getProperty("java.vm.info") + "), " + System.getProperty("java.vm.vendor"));
        log("LWJGL: " + Sys.getVersion());
        openGlVersion = GL11.glGetString(GL11.GL_VERSION);
        openGlRenderer = GL11.glGetString(GL11.GL_RENDERER);
        openGlVendor = GL11.glGetString(GL11.GL_VENDOR);
        log("OpenGL: " + openGlRenderer + ", version " + openGlVersion + ", " + openGlVendor);
        log("OpenGL Version: " + getOpenGlVersionString());

        if (!GLContext.getCapabilities().OpenGL12) {
            log("OpenGL Mipmap levels: Not available (GL12.GL_TEXTURE_MAX_LEVEL)");
        }

        fancyFogAvailable = GLContext.getCapabilities().GL_NV_fog_distance;

        if (!fancyFogAvailable) {
            log("OpenGL Fancy fog: Not available (GL_NV_fog_distance)");
        }

        occlusionAvailable = GLContext.getCapabilities().GL_ARB_occlusion_query;

        if (!occlusionAvailable) {
            log("OpenGL Occlussion culling: Not available (GL_ARB_occlusion_query)");
        }

        int i = TextureUtils.getGLMaximumTextureSize();
        dbg("Maximum texture size: " + i + "x" + i);
    }

    public static String getBuild() {
        if (build == null) {
            try {
                InputStream inputstream = Config.class.getResourceAsStream("/buildof.txt");

                if (inputstream == null) {
                    return null;
                }

                build = readLines(inputstream)[0];
            } catch (Exception exception) {
                warn(exception.getClass().getName() + ": " + exception.getMessage());
                build = "";
            }
        }

        return build;
    }

    public static boolean isFancyFogAvailable() {
        return fancyFogAvailable;
    }

    public static int getMinecraftVersionInt() {
        if (minecraftVersionInt < 0) {
            String[] astring = tokenize("1.8.9", ".");
            int i = 0;

            if (astring.length > 0) {
                i += 10000 * parseInt(astring[0], 0);
            }

            if (astring.length > 1) {
                i += 100 * parseInt(astring[1], 0);
            }

            if (astring.length > 2) {
                i += parseInt(astring[2], 0);
            }

            minecraftVersionInt = i;
        }

        return minecraftVersionInt;
    }

    public static String getOpenGlVersionString() {
        GlVersion glversion = getGlVersion();
        return glversion.getMajor() + "." + glversion.getMinor() + "." + glversion.getRelease();
    }

    private static GlVersion getGlVersionLwjgl() {
        return GLContext.getCapabilities().OpenGL44 ? new GlVersion(4, 4) : (GLContext.getCapabilities().OpenGL43 ? new GlVersion(4, 3) : (GLContext.getCapabilities().OpenGL42 ? new GlVersion(4, 2) : (GLContext.getCapabilities().OpenGL41 ? new GlVersion(4, 1) : (GLContext.getCapabilities().OpenGL40 ? new GlVersion(4, 0) : (GLContext.getCapabilities().OpenGL33 ? new GlVersion(3, 3) : (GLContext.getCapabilities().OpenGL32 ? new GlVersion(3, 2) : (GLContext.getCapabilities().OpenGL31 ? new GlVersion(3, 1) : (GLContext.getCapabilities().OpenGL30 ? new GlVersion(3, 0) : (GLContext.getCapabilities().OpenGL21 ? new GlVersion(2, 1) : (GLContext.getCapabilities().OpenGL20 ? new GlVersion(2, 0) : (GLContext.getCapabilities().OpenGL15 ? new GlVersion(1, 5) : (GLContext.getCapabilities().OpenGL14 ? new GlVersion(1, 4) : (GLContext.getCapabilities().OpenGL13 ? new GlVersion(1, 3) : (GLContext.getCapabilities().OpenGL12 ? new GlVersion(1, 2) : (GLContext.getCapabilities().OpenGL11 ? new GlVersion(1, 1) : new GlVersion(1, 0))))))))))))))));
    }

    public static GlVersion getGlVersion() {
        if (glVersion == null) {
            String s = GL11.glGetString(GL11.GL_VERSION);
            glVersion = parseGlVersion(s, null);

            if (glVersion == null) {
                glVersion = getGlVersionLwjgl();
            }
        }

        return glVersion;
    }

    public static GlVersion getGlslVersion() {
        if (glslVersion == null) {
            String s = GL11.glGetString(GL20.GL_SHADING_LANGUAGE_VERSION);
            glslVersion = parseGlVersion(s, null);

            if (glslVersion == null) {
                glslVersion = new GlVersion(1, 10);
            }
        }

        return glslVersion;
    }

    public static GlVersion parseGlVersion(String p_parseGlVersion_0_, GlVersion p_parseGlVersion_1_) {
        try {
            if (p_parseGlVersion_0_ == null) {
                return p_parseGlVersion_1_;
            } else {
                Pattern pattern = Pattern.compile("([0-9]+)\\.([0-9]+)(\\.([0-9]+))?(.+)?");
                Matcher matcher = pattern.matcher(p_parseGlVersion_0_);

                if (!matcher.matches()) {
                    return p_parseGlVersion_1_;
                } else {
                    int i = Integer.parseInt(matcher.group(1));
                    int j = Integer.parseInt(matcher.group(2));
                    int k = matcher.group(4) != null ? Integer.parseInt(matcher.group(4)) : 0;
                    String s = matcher.group(5);
                    return new GlVersion(i, j, k, s);
                }
            }
        } catch (Exception exception) {
            exception.printStackTrace();
            return p_parseGlVersion_1_;
        }
    }

    public static String[] getOpenGlExtensions() {
        if (openGlExtensions == null) {
            openGlExtensions = detectOpenGlExtensions();
        }

        return openGlExtensions;
    }

    private static String[] detectOpenGlExtensions() {
        try {
            GlVersion glversion = getGlVersion();

            if (glversion.getMajor() >= 3) {
                int i = GL11.glGetInteger(33309);

                if (i > 0) {
                    String[] astring = new String[i];

                    for (int j = 0; j < i; ++j) {
                        astring[j] = GL30.glGetStringi(7939, j);
                    }

                    return astring;
                }
            }
        } catch (Exception exception1) {
            exception1.printStackTrace();
        }

        try {
            String s = GL11.glGetString(GL11.GL_EXTENSIONS);
            return s.split(" ");
        } catch (Exception exception) {
            exception.printStackTrace();
        }

        return new String[0];
    }

    public static void updateThreadPriorities() {
        updateAvailableProcessors();

        if (isSingleProcessor()) {
            if (PerformanceOption.isSmoothWorld()) {
                minecraftThread.setPriority(10);
                setThreadPriority("Server thread", 1);
            } else {
                minecraftThread.setPriority(5);
                setThreadPriority("Server thread", 5);
            }
        } else {
            minecraftThread.setPriority(10);
            setThreadPriority("Server thread", 5);
        }
    }

    private static void setThreadPriority(String p_setThreadPriority_0_, int p_setThreadPriority_1_) {
        try {
            ThreadGroup threadgroup = Thread.currentThread().getThreadGroup();

            if (threadgroup == null) {
                return;
            }

            int i = (threadgroup.activeCount() + 10) * 2;
            Thread[] athread = new Thread[i];
            threadgroup.enumerate(athread, false);

            for (Thread thread : athread) {
                if (thread != null && thread.getName().startsWith(p_setThreadPriority_0_)) {
                    thread.setPriority(p_setThreadPriority_1_);
                }
            }
        } catch (Throwable throwable) {
            warn(throwable.getClass().getName() + ": " + throwable.getMessage());
        }
    }

    public static boolean isMinecraftThread() {
        return Thread.currentThread() == minecraftThread;
    }

    public static void detail(String p_detail_0_) {
        if (logDetail) {
            LOGGER.info("[OptiFine] {}", p_detail_0_);
        }
    }

    public static void dbg(String p_dbg_0_) {
        LOGGER.info("[OptiFine] {}", p_dbg_0_);
    }

    public static void warn(String p_warn_0_) {
        LOGGER.warn("[OptiFine] {}", p_warn_0_);
    }

    public static void error(String p_error_0_) {
        LOGGER.error("[OptiFine] {}", p_error_0_);
    }

    public static void log(String p_log_0_) {
        dbg(p_log_0_);
    }

    public static void updateTexturePackClouds() {
        texturePackClouds = 0;
        IResourceManager iresourcemanager = getResourceManager();

        if (iresourcemanager != null) {
            try {
                InputStream inputstream = iresourcemanager.getResource(new ResourceLocation("mcpatcher/color.properties")).getInputStream();

                if (inputstream == null) {
                    return;
                }

                Properties properties = new PropertiesOrdered();
                properties.load(inputstream);
                inputstream.close();
                String s = properties.getProperty("clouds");

                if (s == null) {
                    return;
                }

                dbg("Texture pack clouds: " + s);
                s = s.toLowerCase();

                if (s.equals("fast")) {
                    texturePackClouds = 1;
                }

                if (s.equals("fancy")) {
                    texturePackClouds = 2;
                }

                if (s.equals("off")) {
                    texturePackClouds = 3;
                }
            } catch (Exception ignored) {
            }
        }
    }

    public static ModelManager getModelManager() {
        return minecraft.getRenderItem().modelManager;
    }

    public static int limit(int p_limit_0_, int p_limit_1_, int p_limit_2_) {
        return p_limit_0_ < p_limit_1_ ? p_limit_1_ : (Math.min(p_limit_0_, p_limit_2_));
    }

    public static float limit(float p_limit_0_, float p_limit_1_, float p_limit_2_) {
        return p_limit_0_ < p_limit_1_ ? p_limit_1_ : (Math.min(p_limit_0_, p_limit_2_));
    }

    public static double limit(double p_limit_0_, double p_limit_2_, double p_limit_4_) {
        return p_limit_0_ < p_limit_2_ ? p_limit_2_ : (Math.min(p_limit_0_, p_limit_4_));
    }

    public static float limitTo1(float p_limitTo1_0_) {
        return p_limitTo1_0_ < 0.0F ? 0.0F : (Math.min(p_limitTo1_0_, 1.0F));
    }
    public static float getAmbientOcclusionLevel() {
        return isShaders() && Shaders.aoLevel >= 0.0F ? Shaders.aoLevel : QualityOption.getSmoothLightingLevel() / 100f;
    }

    public static String listToString(List<String> p_listToString_0_) {
        return listToString(p_listToString_0_, ", ");
    }

    public static String listToString(List<String> p_listToString_0_, String p_listToString_1_) {
        if (p_listToString_0_ == null) {
            return "";
        } else {
            StringBuilder stringbuffer = new StringBuilder(p_listToString_0_.size() * 5);

            for (int i = 0; i < p_listToString_0_.size(); ++i) {
                Object object = p_listToString_0_.get(i);

                if (i > 0) {
                    stringbuffer.append(p_listToString_1_);
                }

                stringbuffer.append(object);
            }

            return stringbuffer.toString();
        }
    }

    public static String arrayToString(Object[] p_arrayToString_0_) {
        return arrayToString(p_arrayToString_0_, ", ");
    }

    public static String arrayToString(Object[] p_arrayToString_0_, String p_arrayToString_1_) {
        if (p_arrayToString_0_ == null) {
            return "";
        } else {
            StringBuilder stringbuffer = new StringBuilder(p_arrayToString_0_.length * 5);

            for (int i = 0; i < p_arrayToString_0_.length; ++i) {
                Object object = p_arrayToString_0_[i];

                if (i > 0) {
                    stringbuffer.append(p_arrayToString_1_);
                }

                stringbuffer.append(object);
            }

            return stringbuffer.toString();
        }
    }

    public static String arrayToString(int[] p_arrayToString_0_) {
        return arrayToString(p_arrayToString_0_, ", ");
    }

    public static String arrayToString(int[] p_arrayToString_0_, String p_arrayToString_1_) {
        if (p_arrayToString_0_ == null) {
            return "";
        } else {
            StringBuilder stringbuffer = new StringBuilder(p_arrayToString_0_.length * 5);

            for (int i = 0; i < p_arrayToString_0_.length; ++i) {
                int j = p_arrayToString_0_[i];

                if (i > 0) {
                    stringbuffer.append(p_arrayToString_1_);
                }

                stringbuffer.append(j);
            }

            return stringbuffer.toString();
        }
    }

    public static String arrayToString(float[] p_arrayToString_0_, String p_arrayToString_1_) {
        if (p_arrayToString_0_ == null) {
            return "";
        } else {
            StringBuilder stringbuffer = new StringBuilder(p_arrayToString_0_.length * 5);

            for (int i = 0; i < p_arrayToString_0_.length; ++i) {
                float f = p_arrayToString_0_[i];

                if (i > 0) {
                    stringbuffer.append(p_arrayToString_1_);
                }

                stringbuffer.append(f);
            }

            return stringbuffer.toString();
        }
    }

    public static Minecraft getMinecraft() {
        return minecraft;
    }

    public static TextureManager getTextureManager() {
        return minecraft.getTextureManager();
    }

    public static IResourceManager getResourceManager() {
        return minecraft.getResourceManager();
    }

    public static InputStream getResourceStream(ResourceLocation p_getResourceStream_0_) throws IOException {
        return getResourceStream(minecraft.getResourceManager(), p_getResourceStream_0_);
    }

    public static InputStream getResourceStream(IResourceManager p_getResourceStream_0_, ResourceLocation p_getResourceStream_1_) throws IOException {
        IResource iresource = p_getResourceStream_0_.getResource(p_getResourceStream_1_);
        return iresource == null ? null : iresource.getInputStream();
    }

    public static IResource getResource(ResourceLocation p_getResource_0_) throws IOException {
        return minecraft.getResourceManager().getResource(p_getResource_0_);
    }

    public static boolean hasResource(ResourceLocation p_hasResource_0_) {
        if (p_hasResource_0_ == null) {
            return false;
        } else {
            IResourcePack iresourcepack = getDefiningResourcePack(p_hasResource_0_);
            return iresourcepack != null;
        }
    }

    public static boolean hasResource(IResourceManager p_hasResource_0_, ResourceLocation p_hasResource_1_) {
        try {
            IResource iresource = p_hasResource_0_.getResource(p_hasResource_1_);
            return iresource != null;
        } catch (IOException var3) {
            return false;
        }
    }

    public static IResourcePack[] getResourcePacks() {
        ResourcePackRepository resourcepackrepository = minecraft.getResourcePackRepository();
        List<ResourcePackRepository.Entry> list = resourcepackrepository.getRepositoryEntries();
        List<IResourcePack> list1 = new ArrayList<>();

        for (ResourcePackRepository.Entry o : list) {
            list1.add(o.getResourcePack());
        }

        if (resourcepackrepository.getResourcePackInstance() != null) {
            list1.add(resourcepackrepository.getResourcePackInstance());
        }

        return list1.toArray(new IResourcePack[0]);
    }

    public static String getResourcePackNames() {
        if (minecraft.getResourcePackRepository() == null) {
            return "";
        } else {
            IResourcePack[] airesourcepack = getResourcePacks();

            if (airesourcepack.length == 0) {
                return getDefaultResourcePack().getPackName();
            } else {
                String[] astring = new String[airesourcepack.length];

                for (int i = 0; i < airesourcepack.length; ++i) {
                    astring[i] = airesourcepack[i].getPackName();
                }

                return arrayToString(astring);
            }
        }
    }

    public static DefaultResourcePack getDefaultResourcePack() {
        if (defaultResourcePackLazy == null) {
            Minecraft minecraft = Minecraft.getMinecraft();
            defaultResourcePackLazy = minecraft.mcDefaultResourcePack;

            if (defaultResourcePackLazy == null) {
                ResourcePackRepository resourcepackrepository = minecraft.getResourcePackRepository();

                if (resourcepackrepository != null) {
                    defaultResourcePackLazy = (DefaultResourcePack) resourcepackrepository.rprDefaultResourcePack;
                }
            }
        }

        return defaultResourcePackLazy;
    }

    public static boolean isFromDefaultResourcePack(ResourceLocation p_isFromDefaultResourcePack_0_) {
        IResourcePack iresourcepack = getDefiningResourcePack(p_isFromDefaultResourcePack_0_);
        return iresourcepack == getDefaultResourcePack();
    }

    public static IResourcePack getDefiningResourcePack(ResourceLocation p_getDefiningResourcePack_0_) {
        ResourcePackRepository resourcepackrepository = minecraft.getResourcePackRepository();
        IResourcePack iresourcepack = resourcepackrepository.getResourcePackInstance();

        if (iresourcepack != null && iresourcepack.resourceExists(p_getDefiningResourcePack_0_)) {
            return iresourcepack;
        } else {
            List<ResourcePackRepository.Entry> list = resourcepackrepository.repositoryEntries;

            for (int i = list.size() - 1; i >= 0; --i) {
                ResourcePackRepository.Entry resourcepackrepository$entry = list.get(i);
                IResourcePack iresourcepack1 = resourcepackrepository$entry.getResourcePack();

                if (iresourcepack1.resourceExists(p_getDefiningResourcePack_0_)) {
                    return iresourcepack1;
                }
            }

            if (getDefaultResourcePack().resourceExists(p_getDefiningResourcePack_0_)) {
                return getDefaultResourcePack();
            } else {
                return null;
            }
        }
    }

    public static RenderGlobal getRenderGlobal() {
        return minecraft.renderGlobal;
    }

    public static void sleep(long p_sleep_0_) {
        try {
            Thread.sleep(p_sleep_0_);
        } catch (InterruptedException interruptedexception) {
            interruptedexception.printStackTrace();
        }
    }

    public static boolean isMultiTexture() {
        return QualityOption.getAnisotropicFilteringLevel() > 1;
    }

    public static boolean between(int p_between_0_, int p_between_1_, int p_between_2_) {
        return p_between_0_ >= p_between_1_ && p_between_0_ <= p_between_2_;
    }

    public static boolean between(float p_between_0_, float p_between_1_, float p_between_2_) {
        return p_between_0_ >= p_between_1_ && p_between_0_ <= p_between_2_;
    }

    public static Dimension getFullscreenDimension() {
        if (desktopDisplayMode == null) {
            return null;
        } else {
            String s = OtherOption.getFullscreenMode();

            if (s.equals("Default")) {
                return new Dimension(desktopDisplayMode.getWidth(), desktopDisplayMode.getHeight());
            } else {
                String[] astring = tokenize(s, " x");
                return astring.length < 2 ? new Dimension(desktopDisplayMode.getWidth(), desktopDisplayMode.getHeight()) : new Dimension(parseInt(astring[0], -1), parseInt(astring[1], -1));
            }
        }
    }

    public static int parseInt(String p_parseInt_0_, int p_parseInt_1_) {
        try {
            if (p_parseInt_0_ == null) {
                return p_parseInt_1_;
            } else {
                p_parseInt_0_ = p_parseInt_0_.trim();
                return Integer.parseInt(p_parseInt_0_);
            }
        } catch (NumberFormatException var3) {
            return p_parseInt_1_;
        }
    }

    public static float parseFloat(String p_parseFloat_0_, float p_parseFloat_1_) {
        try {
            if (p_parseFloat_0_ == null) {
                return p_parseFloat_1_;
            } else {
                p_parseFloat_0_ = p_parseFloat_0_.trim();
                return Float.parseFloat(p_parseFloat_0_);
            }
        } catch (NumberFormatException var3) {
            return p_parseFloat_1_;
        }
    }

    public static boolean parseBoolean(String p_parseBoolean_0_, boolean p_parseBoolean_1_) {
        try {
            if (p_parseBoolean_0_ == null) {
                return p_parseBoolean_1_;
            } else {
                p_parseBoolean_0_ = p_parseBoolean_0_.trim();
                return Boolean.parseBoolean(p_parseBoolean_0_);
            }
        } catch (NumberFormatException var3) {
            return p_parseBoolean_1_;
        }
    }

    public static Boolean parseBoolean(String p_parseBoolean_0_, Boolean p_parseBoolean_1_) {
        try {
            if (p_parseBoolean_0_ == null) {
                return p_parseBoolean_1_;
            } else {
                p_parseBoolean_0_ = p_parseBoolean_0_.trim().toLowerCase();
                return p_parseBoolean_0_.equals("true") || (!p_parseBoolean_0_.equals("false") && p_parseBoolean_1_);
            }
        } catch (NumberFormatException var3) {
            return p_parseBoolean_1_;
        }
    }

    public static String[] tokenize(String p_tokenize_0_, String p_tokenize_1_) {
        StringTokenizer stringtokenizer = new StringTokenizer(p_tokenize_0_, p_tokenize_1_);
        List<String> list = new ArrayList<>();

        while (stringtokenizer.hasMoreTokens()) {
            String s = stringtokenizer.nextToken();
            list.add(s);
        }

        return list.toArray(new String[0]);
    }

    public static DisplayMode[] getDisplayModes() {
        if (displayModes == null) {
            try {
                DisplayMode[] adisplaymode = Display.getAvailableDisplayModes();
                Set<Dimension> set = getDisplayModeDimensions(adisplaymode);
                List<DisplayMode> list = new ArrayList<>();

                for (Dimension dimension : set) {
                    DisplayMode[] adisplaymode1 = getDisplayModes(adisplaymode, dimension);
                    DisplayMode displaymode = getDisplayMode(adisplaymode1, desktopDisplayMode);

                    if (displaymode != null) {
                        list.add(displaymode);
                    }
                }

                DisplayMode[] adisplaymode2 = list.toArray(new DisplayMode[0]);
                Arrays.sort(adisplaymode2, new DisplayModeComparator());
                return adisplaymode2;
            } catch (Exception exception) {
                exception.printStackTrace();
                displayModes = new DisplayMode[]{ desktopDisplayMode };
            }
        }

        return displayModes;
    }

    public static DisplayMode getLargestDisplayMode() {
        DisplayMode[] adisplaymode = getDisplayModes();

        if (adisplaymode != null && adisplaymode.length >= 1) {
            DisplayMode displaymode = adisplaymode[adisplaymode.length - 1];
            return desktopDisplayMode.getWidth() > displaymode.getWidth() ? desktopDisplayMode : (desktopDisplayMode.getWidth() == displaymode.getWidth() && desktopDisplayMode.getHeight() > displaymode.getHeight() ? desktopDisplayMode : displaymode);
        } else {
            return desktopDisplayMode;
        }
    }

    private static Set<Dimension> getDisplayModeDimensions(DisplayMode[] p_getDisplayModeDimensions_0_) {
        Set<Dimension> set = new HashSet<>();

        for (DisplayMode displaymode : p_getDisplayModeDimensions_0_) {
            Dimension dimension = new Dimension(displaymode.getWidth(), displaymode.getHeight());
            set.add(dimension);
        }

        return set;
    }

    private static DisplayMode[] getDisplayModes(DisplayMode[] p_getDisplayModes_0_, Dimension p_getDisplayModes_1_) {
        List<DisplayMode> list = new ArrayList<>();

        for (DisplayMode displaymode : p_getDisplayModes_0_) {
            if ((double) displaymode.getWidth() == p_getDisplayModes_1_.getWidth() && (double) displaymode.getHeight() == p_getDisplayModes_1_.getHeight()) {
                list.add(displaymode);
            }
        }

        return list.toArray(new DisplayMode[0]);
    }

    private static DisplayMode getDisplayMode(DisplayMode[] p_getDisplayMode_0_, DisplayMode p_getDisplayMode_1_) {
        if (p_getDisplayMode_1_ != null) {
            for (DisplayMode displaymode : p_getDisplayMode_0_) {
                if (displaymode.getBitsPerPixel() == p_getDisplayMode_1_.getBitsPerPixel() && displaymode.getFrequency() == p_getDisplayMode_1_.getFrequency()) {
                    return displaymode;
                }
            }
        }

        if (p_getDisplayMode_0_.length == 0) {
            return null;
        } else {
            Arrays.sort(p_getDisplayMode_0_, new DisplayModeComparator());
            return p_getDisplayMode_0_[p_getDisplayMode_0_.length - 1];
        }
    }

    public static String[] getDisplayModeNames() {
        DisplayMode[] adisplaymode = getDisplayModes();
        String[] astring = new String[adisplaymode.length];

        for (int i = 0; i < adisplaymode.length; ++i) {
            DisplayMode displaymode = adisplaymode[i];
            String s = displaymode.getWidth() + "x" + displaymode.getHeight();
            astring[i] = s;
        }

        return astring;
    }

    public static DisplayMode getDisplayMode(Dimension p_getDisplayMode_0_) {
        DisplayMode[] adisplaymode = getDisplayModes();

        for (DisplayMode displaymode : adisplaymode) {
            if (displaymode.getWidth() == p_getDisplayMode_0_.width && displaymode.getHeight() == p_getDisplayMode_0_.height) {
                return displaymode;
            }
        }

        return desktopDisplayMode;
    }

    public static void checkGlError(String p_checkGlError_0_) {
        int i = GlStateManager.glGetError();

        if (i != 0 && GlErrors.isEnabled(i)) {
            String s = getGlErrorString(i);
            String s1 = String.format("OpenGL error: %s (%s), at: %s", i, s, p_checkGlError_0_);
            error(s1);

            if (DebuggingOption.isShowGlErrors() && TimedEvent.isActive("ShowGlError", 10000L)) {
                String s2 = LanguageManager.format("of.message.openglError", i, s);
                minecraft.ingameGUI.getChatGUI().printChatMessage(new ChatComponentText(s2));
            }
        }
    }

    public static boolean isShaders() {
        return Shaders.shaderPackLoaded;
    }

    public static String[] readLines(InputStream p_readLines_0_) throws IOException {
        List<String> list = new ArrayList<>();
        InputStreamReader inputstreamreader = new InputStreamReader(p_readLines_0_, StandardCharsets.US_ASCII);
        BufferedReader bufferedreader = new BufferedReader(inputstreamreader);

        while (true) {
            String s = bufferedreader.readLine();

            if (s == null) {
                return list.toArray(new String[0]);
            }

            list.add(s);
        }
    }

    public static String readInputStream(InputStream p_readInputStream_0_) throws IOException {
        return readInputStream(p_readInputStream_0_, "ASCII");
    }

    public static String readInputStream(InputStream p_readInputStream_0_, String p_readInputStream_1_) throws IOException {
        InputStreamReader inputstreamreader = new InputStreamReader(p_readInputStream_0_, p_readInputStream_1_);
        BufferedReader bufferedreader = new BufferedReader(inputstreamreader);
        StringBuilder stringbuffer = new StringBuilder();

        while (true) {
            String s = bufferedreader.readLine();

            if (s == null) {
                return stringbuffer.toString();
            }

            stringbuffer.append(s);
            stringbuffer.append("\n");
        }
    }

    public static byte[] readAll(InputStream p_readAll_0_) throws IOException {
        ByteArrayOutputStream bytearrayoutputstream = new ByteArrayOutputStream();
        byte[] abyte = new byte[1024];

        while (true) {
            int i = p_readAll_0_.read(abyte);

            if (i < 0) {
                p_readAll_0_.close();
                return bytearrayoutputstream.toByteArray();
            }

            bytearrayoutputstream.write(abyte, 0, i);
        }
    }

    private static String[] splitRelease(String p_splitRelease_0_) {
        if (p_splitRelease_0_ != null && !p_splitRelease_0_.isEmpty()) {
            Pattern pattern = Pattern.compile("([A-Z])([0-9]+)(.*)");
            Matcher matcher = pattern.matcher(p_splitRelease_0_);

            if (!matcher.matches()) {
                return new String[]{"", "", ""};
            } else {
                String s = normalize(matcher.group(1));
                String s1 = normalize(matcher.group(2));
                String s2 = normalize(matcher.group(3));
                return new String[]{s, s1, s2};
            }
        } else {
            return new String[]{"", "", ""};
        }
    }

    public static int intHash(int p_intHash_0_) {
        p_intHash_0_ = p_intHash_0_ ^ 61 ^ p_intHash_0_ >> 16;
        p_intHash_0_ = p_intHash_0_ + (p_intHash_0_ << 3);
        p_intHash_0_ = p_intHash_0_ ^ p_intHash_0_ >> 4;
        p_intHash_0_ = p_intHash_0_ * 668265261;
        p_intHash_0_ = p_intHash_0_ ^ p_intHash_0_ >> 15;
        return p_intHash_0_;
    }

    public static int getRandom(BlockPos p_getRandom_0_, int p_getRandom_1_) {
        int i = intHash(p_getRandom_1_ + 37);
        i = intHash(i + p_getRandom_0_.getX());
        i = intHash(i + p_getRandom_0_.getZ());
        i = intHash(i + p_getRandom_0_.getY());
        return i;
    }

    public static int getAvailableProcessors() {
        return availableProcessors;
    }

    public static void updateAvailableProcessors() {
        availableProcessors = Runtime.getRuntime().availableProcessors();
    }

    public static boolean isSingleProcessor() {
        return getAvailableProcessors() <= 1;
    }

    public static boolean equals(Object p_equals_0_, Object p_equals_1_) {
        return Objects.equals(p_equals_0_, p_equals_1_);
    }

    public static boolean equalsOne(Object p_equalsOne_0_, Object[] p_equalsOne_1_) {
        if (p_equalsOne_1_ != null) {
            for (Object object : p_equalsOne_1_) {
                if (equals(p_equalsOne_0_, object)) {
                    return true;
                }
            }

        }
        return false;
    }

    public static boolean equalsOne(int p_equalsOne_0_, int[] p_equalsOne_1_) {
        for (int j : p_equalsOne_1_) {
            if (j == p_equalsOne_0_) {
                return true;
            }
        }

        return false;
    }

    public static boolean isSameOne(Object p_isSameOne_0_, Object[] p_isSameOne_1_) {
        if (p_isSameOne_1_ != null) {
            for (Object object : p_isSameOne_1_) {
                if (p_isSameOne_0_ == object) {
                    return true;
                }
            }

        }
        return false;
    }

    public static String normalize(String p_normalize_0_) {
        return p_normalize_0_ == null ? "" : p_normalize_0_;
    }

    private static ByteBuffer readIconImage(InputStream p_readIconImage_0_) throws IOException {
        BufferedImage bufferedimage = ImageIO.read(p_readIconImage_0_);
        int[] aint = bufferedimage.getRGB(0, 0, bufferedimage.getWidth(), bufferedimage.getHeight(), null, 0, bufferedimage.getWidth());
        ByteBuffer bytebuffer = ByteBuffer.allocate(4 * aint.length);

        for (int i : aint) {
            bytebuffer.putInt(i << 8 | i >> 24 & 255);
        }

        bytebuffer.flip();
        return bytebuffer;
    }

    public static void checkDisplayMode() {
        try {
            if (minecraft.isFullScreen()) {
                if (fullscreenModeChecked) {
                    return;
                }

                fullscreenModeChecked = true;
                desktopModeChecked = false;
                DisplayMode displaymode = Display.getDisplayMode();
                Dimension dimension = getFullscreenDimension();

                if (dimension == null) {
                    return;
                }

                if (displaymode.getWidth() == dimension.width && displaymode.getHeight() == dimension.height) {
                    return;
                }

                DisplayMode displaymode1 = getDisplayMode(dimension);

                if (displaymode1 == null) {
                    return;
                }

                Display.setDisplayMode(displaymode1);
                minecraft.displayWidth = Display.getDisplayMode().getWidth();
                minecraft.displayHeight = Display.getDisplayMode().getHeight();

                if (minecraft.displayWidth <= 0) {
                    minecraft.displayWidth = 1;
                }

                if (minecraft.displayHeight <= 0) {
                    minecraft.displayHeight = 1;
                }

                if (minecraft.currentScreen != null) {
                    ScaledResolution scaledresolution = new ScaledResolution(minecraft);
                    int i = scaledresolution.getScaledWidth();
                    int j = scaledresolution.getScaledHeight();
                    minecraft.currentScreen.setWorldAndResolution(minecraft, i, j);
                }

                updateFramebufferSize();
                Display.setFullscreen(true);
                Display.setVSyncEnabled(GeneralOption.getVsync());
                GlStateManager.enableTexture2D();
            } else {
                if (desktopModeChecked) {
                    return;
                }

                desktopModeChecked = true;
                fullscreenModeChecked = false;
                Display.setVSyncEnabled(GeneralOption.getVsync());
                Display.update();
                GlStateManager.enableTexture2D();

                if (OSUtil.getOSType() == OSUtil.EnumOS.WINDOWS) {
                    Display.setResizable(false);
                    Display.setResizable(true);
                }
            }
        } catch (Exception exception) {
            exception.printStackTrace();
            OtherOption.getFullscreenModeObject().setValue("Default");
            OtherOption.getFullscreenModeObject().syncValue();
        }
    }

    public static void updateFramebufferSize() {
        minecraft.getFramebuffer().createBindFramebuffer(minecraft.displayWidth, minecraft.displayHeight);

        if (minecraft.entityRenderer != null) {
            minecraft.entityRenderer.updateShaderGroupSize(minecraft.displayWidth, minecraft.displayHeight);
        }

        minecraft.loadingScreen = new LoadingScreenRenderer(minecraft);
    }

    public static Object[] addObjectToArray(Object[] p_addObjectToArray_0_, Object p_addObjectToArray_1_) {
        if (p_addObjectToArray_0_ == null) {
            throw new NullPointerException("The given array is NULL");
        } else {
            int i = p_addObjectToArray_0_.length;
            int j = i + 1;
            Object[] aobject = (Object[]) Array.newInstance(p_addObjectToArray_0_.getClass().getComponentType(), j);
            System.arraycopy(p_addObjectToArray_0_, 0, aobject, 0, i);
            aobject[i] = p_addObjectToArray_1_;
            return aobject;
        }
    }

    public static Object[] addObjectToArray(Object[] p_addObjectToArray_0_, Object p_addObjectToArray_1_, int p_addObjectToArray_2_) {
        List<Object> list = new ArrayList<>(Arrays.asList(p_addObjectToArray_0_));
        list.add(p_addObjectToArray_2_, p_addObjectToArray_1_);
        Object[] aobject = (Object[]) Array.newInstance(p_addObjectToArray_0_.getClass().getComponentType(), list.size());
        return list.toArray(aobject);
    }

    public static Object[] addObjectsToArray(Object[] p_addObjectsToArray_0_, Object[] p_addObjectsToArray_1_) {
        if (p_addObjectsToArray_0_ == null) {
            throw new NullPointerException("The given array is NULL");
        } else if (p_addObjectsToArray_1_.length == 0) {
            return p_addObjectsToArray_0_;
        } else {
            int i = p_addObjectsToArray_0_.length;
            int j = i + p_addObjectsToArray_1_.length;
            Object[] aobject = (Object[]) Array.newInstance(p_addObjectsToArray_0_.getClass().getComponentType(), j);
            System.arraycopy(p_addObjectsToArray_0_, 0, aobject, 0, i);
            System.arraycopy(p_addObjectsToArray_1_, 0, aobject, i, p_addObjectsToArray_1_.length);
            return aobject;
        }
    }

    public static void drawFps() {
        int i = Minecraft.getDebugFPS();
        String s = getUpdates(minecraft.debug);
        int j = minecraft.renderGlobal.getCountActiveRenderers();
        int k = minecraft.renderGlobal.getCountEntitiesRendered();
        int l = minecraft.renderGlobal.getCountTileEntitiesRendered();

        String fps = (DebuggingOption.isRenderNormalFPSCounter()) ? i + " fps" : i + "/" + getFpsMin() + " fps";

        String s1 = fps + ", C: " + j + ", E: " + k + "+" + l + ", U: " + s;
        minecraft.minecraftFontRendererObj.drawString(s1, 2, 2, -2039584);
    }

    public static int getFpsMin() {
        if (!minecraft.debug.equals(mcDebugLast)) {
            mcDebugLast = minecraft.debug;
            FrameTimer frametimer = minecraft.getFrameTimer();
            long[] along = frametimer.getFrames();
            int i = frametimer.getIndex();
            int j = frametimer.getLastIndex();

            if (i != j) {
                int k = Minecraft.getDebugFPS();

                if (k <= 0) {
                    k = 1;
                }

                long i1 = (long) (1.0D / (double) k * 1.0E9D);
                long j1 = 0L;

                for (int k1 = MathHelper.normalizeAngle(i - 1, along.length); k1 != j && (double) j1 < 1.0E9D; k1 = MathHelper.normalizeAngle(k1 - 1, along.length)) {
                    long l1 = along[k1];

                    if (l1 > i1) {
                        i1 = l1;
                    }

                    j1 += l1;
                }

                double d0 = (double) i1 / 1.0E9D;
                fpsMinLast = (int) (1.0D / d0);
            }
        }
        return fpsMinLast;
    }

    private static String getUpdates(String p_getUpdates_0_) {
        int i = p_getUpdates_0_.indexOf(40);

        if (i < 0) {
            return "";
        } else {
            int j = p_getUpdates_0_.indexOf(32, i);
            return j < 0 ? "" : p_getUpdates_0_.substring(i + 1, j);
        }
    }

    public static int getBitsOs() {
        String s = System.getenv("ProgramFiles(X86)");
        return s != null ? 64 : 32;
    }

    public static int getBitsJre() {
        String[] astring = new String[]{"sun.arch.data.model", "com.ibm.vm.bitmode", "os.arch"};

        for (String s : astring) {
            String s1 = System.getProperty(s);

            if (s1 != null && s1.contains("64")) {
                return 64;
            }
        }

        return 32;
    }

    public static boolean isNotify64BitJava() {
        return notify64BitJava;
    }

    public static void setNotify64BitJava(boolean p_setNotify64BitJava_0_) {
        notify64BitJava = p_setNotify64BitJava_0_;
    }

    public static int[] addIntToArray(int[] p_addIntToArray_0_, int p_addIntToArray_1_) {
        return addIntsToArray(p_addIntToArray_0_, new int[]{p_addIntToArray_1_});
    }

    public static int[] addIntsToArray(int[] p_addIntsToArray_0_, int[] p_addIntsToArray_1_) {
        if (p_addIntsToArray_0_ != null && p_addIntsToArray_1_ != null) {
            int i = p_addIntsToArray_0_.length;
            int j = i + p_addIntsToArray_1_.length;
            int[] aint = new int[j];
            System.arraycopy(p_addIntsToArray_0_, 0, aint, 0, i);

            System.arraycopy(p_addIntsToArray_1_, 0, aint, i, p_addIntsToArray_1_.length);

            return aint;
        } else {
            throw new NullPointerException("The given array is NULL");
        }
    }

    public static DynamicTexture getMojangLogoTexture(DynamicTexture p_getMojangLogoTexture_0_) {
        try {
            ResourceLocation resourcelocation = new ResourceLocation("textures/gui/title/mojang.png");
            InputStream inputstream = getResourceStream(resourcelocation);

            if (inputstream == null) {
                return p_getMojangLogoTexture_0_;
            } else {
                BufferedImage bufferedimage = ImageIO.read(inputstream);

                if (bufferedimage == null) {
                    return p_getMojangLogoTexture_0_;
                } else {
                    return new DynamicTexture(bufferedimage);
                }
            }
        } catch (Exception exception) {
            warn(exception.getClass().getName() + ": " + exception.getMessage());
            return p_getMojangLogoTexture_0_;
        }
    }

    public static void writeFile(File p_writeFile_0_, String p_writeFile_1_) throws IOException {
        FileOutputStream fileoutputstream = new FileOutputStream(p_writeFile_0_);
        byte[] abyte = p_writeFile_1_.getBytes(StandardCharsets.US_ASCII);
        fileoutputstream.write(abyte);
        fileoutputstream.close();
    }

    public static TextureMap getTextureMap() {
        return getMinecraft().getTextureMapBlocks();
    }

    public static int[] toPrimitive(Integer[] p_toPrimitive_0_) {
        if (p_toPrimitive_0_ == null) {
            return null;
        } else if (p_toPrimitive_0_.length == 0) {
            return new int[0];
        } else {
            int[] aint = new int[p_toPrimitive_0_.length];

            for (int i = 0; i < aint.length; ++i) {
                aint[i] = p_toPrimitive_0_[i];
            }

            return aint;
        }
    }

    public static boolean isVbo() {
        return OpenGlHelper.useVbo();
    }

    public static String arrayToString(boolean[] p_arrayToString_0_, String p_arrayToString_1_) {
        if (p_arrayToString_0_ == null) {
            return "";
        } else {
            StringBuilder stringbuffer = new StringBuilder(p_arrayToString_0_.length * 5);

            for (int i = 0; i < p_arrayToString_0_.length; ++i) {
                boolean flag = p_arrayToString_0_[i];

                if (i > 0) {
                    stringbuffer.append(p_arrayToString_1_);
                }

                stringbuffer.append(flag);
            }

            return stringbuffer.toString();
        }
    }

    public static boolean isIntegratedServerRunning() {
        return minecraft.getIntegratedServer() != null && minecraft.isIntegratedServerRunning();
    }

    public static IntBuffer createDirectIntBuffer(int p_createDirectIntBuffer_0_) {
        return GLAllocation.createDirectByteBuffer(p_createDirectIntBuffer_0_ << 2).asIntBuffer();
    }

    public static String getGlErrorString(int p_getGlErrorString_0_) {
        return switch (p_getGlErrorString_0_) {
            case 0 -> "No error";
            case 1280 -> "Invalid enum";
            case 1281 -> "Invalid value";
            case 1282 -> "Invalid operation";
            case 1283 -> "Stack overflow";
            case 1284 -> "Stack underflow";
            case 1285 -> "Out of memory";
            case 1286 -> "Invalid framebuffer operation";
            default -> "Unknown";
        };
    }

    public static boolean isTrue(Boolean p_isTrue_0_) {
        return p_isTrue_0_ != null && p_isTrue_0_;
    }

    public static boolean isQuadsToTriangles() {
        return isShaders() && !Shaders.canRenderQuads();
    }
}
