package net.minecraft.server.management;

import com.google.gson.JsonObject;
import com.mojang.authlib.GameProfile;

import java.util.UUID;

public class UserListOpsEntry extends UserListEntry<GameProfile> {
    private final int permissionLevel;
    private final boolean bypassesPlayerLimit;

    public UserListOpsEntry(GameProfile player, int permissionLevelIn, boolean bypassesPlayerLimitIn) {
        super(player);
        this.permissionLevel = permissionLevelIn;
        this.bypassesPlayerLimit = bypassesPlayerLimitIn;
    }

    public UserListOpsEntry(JsonObject jsonObj) {
        super(constructProfile(jsonObj), jsonObj);
        this.permissionLevel = jsonObj.has("level") ? jsonObj.get("level").getAsInt() : 0;
        this.bypassesPlayerLimit = jsonObj.has("bypassesPlayerLimit") && jsonObj.get("bypassesPlayerLimit").getAsBoolean();
    }

    private static GameProfile constructProfile(JsonObject jsonObj) {
        if (jsonObj.has("uuid") && jsonObj.has("name")) {
            String s = jsonObj.get("uuid").getAsString();
            UUID uuid;

            try {
                uuid = UUID.fromString(s);
            } catch (Throwable var4) {
                return null;
            }

            return new GameProfile(uuid, jsonObj.get("name").getAsString());
        } else {
            return null;
        }
    }

    public int getPermissionLevel() {
        return this.permissionLevel;
    }

    public boolean bypassesPlayerLimit() {
        return this.bypassesPlayerLimit;
    }

    protected void onSerialization(JsonObject data) {
        if (this.getValue() != null) {
            data.addProperty("uuid", this.getValue().getId() == null ? "" : this.getValue().getId().toString());
            data.addProperty("name", this.getValue().getName());
            super.onSerialization(data);
            data.addProperty("level", this.permissionLevel);
            data.addProperty("bypassesPlayerLimit", this.bypassesPlayerLimit);
        }
    }
}
