package net.minecraft.scoreboard;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.network.Packet;
import net.minecraft.network.play.server.S3BPacketScoreboardObjective;
import net.minecraft.network.play.server.S3CPacketUpdateScore;
import net.minecraft.network.play.server.S3DPacketDisplayScoreboard;
import net.minecraft.network.play.server.S3EPacketTeams;
import net.minecraft.server.MinecraftServer;

import java.util.Collections;
import java.util.List;
import java.util.Set;

public class ServerScoreboard extends Scoreboard {
    private final MinecraftServer scoreboardMCServer;
    private final Set<ScoreObjective> addedObjectives = Sets.newHashSet();
    private ScoreboardSaveData scoreboardSaveData;

    public ServerScoreboard(MinecraftServer mcServer) {
        this.scoreboardMCServer = mcServer;
    }

    public void onScoreUpdated(Score scoreIn) {
        super.onScoreUpdated(scoreIn);

        if (this.addedObjectives.contains(scoreIn.getObjective())) {
            this.scoreboardMCServer.getConfigurationManager().sendPacketToAllPlayers(new S3CPacketUpdateScore(scoreIn));
        }

        this.markSaveDataDirty();
    }

    public void broadcastScoreUpdate(String scoreName) {
        super.broadcastScoreUpdate(scoreName);
        this.scoreboardMCServer.getConfigurationManager().sendPacketToAllPlayers(new S3CPacketUpdateScore(scoreName));
        this.markSaveDataDirty();
    }

    public void broadcastScoreUpdate(String scoreName, ScoreObjective objective) {
        super.broadcastScoreUpdate(scoreName, objective);
        this.scoreboardMCServer.getConfigurationManager().sendPacketToAllPlayers(new S3CPacketUpdateScore(scoreName, objective));
        this.markSaveDataDirty();
    }

    public void setObjectiveInDisplaySlot(int objectiveSlot, ScoreObjective objective) {
        ScoreObjective scoreobjective = this.getObjectiveInDisplaySlot(objectiveSlot);
        super.setObjectiveInDisplaySlot(objectiveSlot, objective);

        if (scoreobjective != objective && scoreobjective != null) {
            if (this.getObjectiveDisplaySlotCount(scoreobjective) > 0) {
                this.scoreboardMCServer.getConfigurationManager().sendPacketToAllPlayers(new S3DPacketDisplayScoreboard(objectiveSlot, objective));
            } else {
                this.sendDisplaySlotRemovalPackets(scoreobjective);
            }
        }

        if (objective != null) {
            if (this.addedObjectives.contains(objective)) {
                this.scoreboardMCServer.getConfigurationManager().sendPacketToAllPlayers(new S3DPacketDisplayScoreboard(objectiveSlot, objective));
            } else {
                this.addObjective(objective);
            }
        }

        this.markSaveDataDirty();
    }

    public boolean addPlayerToTeam(String player, String newTeam) {
        if (super.addPlayerToTeam(player, newTeam)) {
            ScorePlayerTeam scoreplayerteam = this.getTeam(newTeam);
            this.scoreboardMCServer.getConfigurationManager().sendPacketToAllPlayers(new S3EPacketTeams(scoreplayerteam, Collections.singletonList(player), 3));
            this.markSaveDataDirty();
            return true;
        } else {
            return false;
        }
    }

    public void removePlayerFromTeam(String username, ScorePlayerTeam playerTeam) {
        super.removePlayerFromTeam(username, playerTeam);
        this.scoreboardMCServer.getConfigurationManager().sendPacketToAllPlayers(new S3EPacketTeams(playerTeam, Collections.singletonList(username), 4));
        this.markSaveDataDirty();
    }

    public void onScoreObjectiveAdded(ScoreObjective scoreObjectiveIn) {
        super.onScoreObjectiveAdded(scoreObjectiveIn);
        this.markSaveDataDirty();
    }

    public void onObjectiveDisplayNameChanged(ScoreObjective objective) {
        super.onObjectiveDisplayNameChanged(objective);

        if (this.addedObjectives.contains(objective)) {
            this.scoreboardMCServer.getConfigurationManager().sendPacketToAllPlayers(new S3BPacketScoreboardObjective(objective, 2));
        }

        this.markSaveDataDirty();
    }

    public void onScoreObjectiveRemoved(ScoreObjective objective) {
        super.onScoreObjectiveRemoved(objective);

        if (this.addedObjectives.contains(objective)) {
            this.sendDisplaySlotRemovalPackets(objective);
        }

        this.markSaveDataDirty();
    }

    public void broadcastTeamCreated(ScorePlayerTeam playerTeam) {
        super.broadcastTeamCreated(playerTeam);
        this.scoreboardMCServer.getConfigurationManager().sendPacketToAllPlayers(new S3EPacketTeams(playerTeam, 0));
        this.markSaveDataDirty();
    }

    public void sendTeamUpdate(ScorePlayerTeam playerTeam) {
        super.sendTeamUpdate(playerTeam);
        this.scoreboardMCServer.getConfigurationManager().sendPacketToAllPlayers(new S3EPacketTeams(playerTeam, 2));
        this.markSaveDataDirty();
    }

    public void broadcastTeamRemove(ScorePlayerTeam playerTeam) {
        super.broadcastTeamRemove(playerTeam);
        this.scoreboardMCServer.getConfigurationManager().sendPacketToAllPlayers(new S3EPacketTeams(playerTeam, 1));
        this.markSaveDataDirty();
    }

    public void setScoreboardSaveData(ScoreboardSaveData data) {
        this.scoreboardSaveData = data;
    }

    protected void markSaveDataDirty() {
        if (this.scoreboardSaveData != null) {
            this.scoreboardSaveData.markDirty();
        }
    }

    public List<Packet<?>> getCreatePackets(ScoreObjective objective) {
        List<Packet<?>> list = Lists.newArrayList();
        list.add(new S3BPacketScoreboardObjective(objective, 0));

        for (int i = 0; i < 19; ++i) {
            if (this.getObjectiveInDisplaySlot(i) == objective) {
                list.add(new S3DPacketDisplayScoreboard(i, objective));
            }
        }

        for (Score score : this.getSortedScores(objective)) {
            list.add(new S3CPacketUpdateScore(score));
        }

        return list;
    }

    public void addObjective(ScoreObjective objective) {
        List<Packet<?>> list = this.getCreatePackets(objective);

        for (EntityPlayerMP entityplayermp : this.scoreboardMCServer.getConfigurationManager().getPlayerList()) {
            for (Packet<?> packet : list) {
                entityplayermp.playerNetServerHandler.sendPacket(packet);
            }
        }

        this.addedObjectives.add(objective);
    }

    public List<Packet<?>> getDestroyPackets(ScoreObjective objective) {
        List<Packet<?>> list = Lists.newArrayList();
        list.add(new S3BPacketScoreboardObjective(objective, 1));

        for (int i = 0; i < 19; ++i) {
            if (this.getObjectiveInDisplaySlot(i) == objective) {
                list.add(new S3DPacketDisplayScoreboard(i, objective));
            }
        }

        return list;
    }

    public void sendDisplaySlotRemovalPackets(ScoreObjective objective) {
        List<Packet<?>> list = this.getDestroyPackets(objective);

        for (EntityPlayerMP entityplayermp : this.scoreboardMCServer.getConfigurationManager().getPlayerList()) {
            for (Packet<?> packet : list) {
                entityplayermp.playerNetServerHandler.sendPacket(packet);
            }
        }

        this.addedObjectives.remove(objective);
    }

    public int getObjectiveDisplaySlotCount(ScoreObjective objective) {
        int i = 0;

        for (int j = 0; j < 19; ++j) {
            if (this.getObjectiveInDisplaySlot(j) == objective) {
                ++i;
            }
        }

        return i;
    }
}