package net.minecraft.scoreboard;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import net.minecraft.entity.Entity;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.util.EnumChatFormatting;

import java.util.Collection;
import java.util.List;
import java.util.Map;

public class Scoreboard {
    private static String[] displaySlots = null;
    private final Map<String, ScoreObjective> scoreObjectives = Maps.newHashMap();
    private final Map<IScoreObjectiveCriteria, List<ScoreObjective>> scoreObjectiveCriterias = Maps.newHashMap();
    private final Map<String, Map<ScoreObjective, Score>> entitiesScoreObjectives = Maps.newHashMap();
    private final ScoreObjective[] objectiveDisplaySlots = new ScoreObjective[19];
    private final Map<String, ScorePlayerTeam> teams = Maps.newHashMap();
    private final Map<String, ScorePlayerTeam> teamMemberships = Maps.newHashMap();

    public static String getObjectiveDisplaySlot(int id) {
        return switch (id) {
            case 0 -> "list";
            case 1 -> "sidebar";
            case 2 -> "belowName";
            default -> {
                if (id >= 3 && id <= 18) {
                    EnumChatFormatting enumchatformatting = EnumChatFormatting.fromColorIndex(id - 3);

                    if (enumchatformatting != null && enumchatformatting != EnumChatFormatting.RESET) {
                        yield "sidebar.team." + enumchatformatting.getFriendlyName();
                    }
                }

                yield null;
            }
        };
    }

    public static int getObjectiveDisplaySlotNumber(String name) {
        if (name.equalsIgnoreCase("list")) {
            return 0;
        } else if (name.equalsIgnoreCase("sidebar")) {
            return 1;
        } else if (name.equalsIgnoreCase("belowName")) {
            return 2;
        } else {
            if (name.startsWith("sidebar.team.")) {
                String s = name.substring("sidebar.team.".length());
                EnumChatFormatting enumchatformatting = EnumChatFormatting.getValueByName(s);

                if (enumchatformatting != null && enumchatformatting.getColorIndex() >= 0) {
                    return enumchatformatting.getColorIndex() + 3;
                }
            }

            return -1;
        }
    }

    public static String[] getDisplaySlotStrings() {
        if (displaySlots == null) {
            displaySlots = new String[19];

            for (int i = 0; i < 19; ++i) {
                displaySlots[i] = getObjectiveDisplaySlot(i);
            }
        }

        return displaySlots;
    }

    public ScoreObjective getObjective(String name) {
        return this.scoreObjectives.get(name);
    }

    public ScoreObjective addScoreObjective(String name, IScoreObjectiveCriteria criteria) {
        if (name.length() > 16) {
            throw new IllegalArgumentException("The objective name '" + name + "' is too long!");
        } else {
            ScoreObjective scoreobjective = this.getObjective(name);

            if (scoreobjective != null) {
                throw new IllegalArgumentException("An objective with the name '" + name + "' already exists!");
            } else {
                scoreobjective = new ScoreObjective(this, name, criteria);
                List<ScoreObjective> list = this.scoreObjectiveCriterias.computeIfAbsent(criteria, k -> Lists.newArrayList());

                list.add(scoreobjective);
                this.scoreObjectives.put(name, scoreobjective);
                this.onScoreObjectiveAdded(scoreobjective);
                return scoreobjective;
            }
        }
    }

    public Collection<ScoreObjective> getObjectivesFromCriteria(IScoreObjectiveCriteria criteria) {
        Collection<ScoreObjective> collection = this.scoreObjectiveCriterias.get(criteria);
        return collection == null ? Lists.newArrayList() : Lists.newArrayList(collection);
    }

    public boolean entityHasObjective(String name, ScoreObjective objective) {
        Map<ScoreObjective, Score> map = this.entitiesScoreObjectives.get(name);

        if (map == null) {
            return false;
        } else {
            Score score = map.get(objective);
            return score != null;
        }
    }

    public Score getValueFromObjective(String name, ScoreObjective objective) {
        if (name.length() > 40) {
            throw new IllegalArgumentException("The player name '" + name + "' is too long!");
        } else {
            Map<ScoreObjective, Score> map = this.entitiesScoreObjectives.computeIfAbsent(name, k -> Maps.newHashMap());

            return map.computeIfAbsent(objective, o -> new Score(this, o, name));
        }
    }

    public Collection<Score> getSortedScores(ScoreObjective objective) {
        List<Score> list = Lists.newArrayList();

        for (Map<ScoreObjective, Score> map : this.entitiesScoreObjectives.values()) {
            Score score = map.get(objective);

            if (score != null) {
                list.add(score);
            }
        }

        list.sort(Score.scoreComparator);
        return list;
    }

    public Collection<ScoreObjective> getScoreObjectives() {
        return this.scoreObjectives.values();
    }

    public Collection<String> getObjectiveNames() {
        return this.entitiesScoreObjectives.keySet();
    }

    public void removeObjectiveFromEntity(String name, ScoreObjective objective) {
        if (objective == null) {
            Map<ScoreObjective, Score> map = this.entitiesScoreObjectives.remove(name);

            if (map != null) {
                this.broadcastScoreUpdate(name);
            }
        } else {
            Map<ScoreObjective, Score> map2 = this.entitiesScoreObjectives.get(name);

            if (map2 != null) {
                Score score = map2.remove(objective);

                if (map2.isEmpty()) {
                    Map<ScoreObjective, Score> map1 = this.entitiesScoreObjectives.remove(name);

                    if (map1 != null) {
                        this.broadcastScoreUpdate(name);
                    }
                } else if (score != null) {
                    this.broadcastScoreUpdate(name, objective);
                }
            }
        }
    }

    public Collection<Score> getScores() {
        Collection<Map<ScoreObjective, Score>> collection = this.entitiesScoreObjectives.values();
        List<Score> list = Lists.newArrayList();

        for (Map<ScoreObjective, Score> map : collection) {
            list.addAll(map.values());
        }

        return list;
    }

    public Map<ScoreObjective, Score> getObjectivesForEntity(String name) {
        Map<ScoreObjective, Score> map = this.entitiesScoreObjectives.get(name);

        if (map == null) {
            map = Maps.newHashMap();
        }

        return map;
    }

    public void removeObjective(ScoreObjective objective) {
        this.scoreObjectives.remove(objective.getName());

        for (int i = 0; i < 19; ++i) {
            if (this.getObjectiveInDisplaySlot(i) == objective) {
                this.setObjectiveInDisplaySlot(i, null);
            }
        }

        List<ScoreObjective> list = this.scoreObjectiveCriterias.get(objective.getCriteria());

        if (list != null) {
            list.remove(objective);
        }

        for (Map<ScoreObjective, Score> map : this.entitiesScoreObjectives.values()) {
            map.remove(objective);
        }

        this.onScoreObjectiveRemoved(objective);
    }

    public void setObjectiveInDisplaySlot(int objectiveSlot, ScoreObjective objective) {
        this.objectiveDisplaySlots[objectiveSlot] = objective;
    }

    public ScoreObjective getObjectiveInDisplaySlot(int slotIn) {
        return this.objectiveDisplaySlots[slotIn];
    }

    public ScorePlayerTeam getTeam(String teamName) {
        return this.teams.get(teamName);
    }

    public ScorePlayerTeam createTeam(String name) {
        if (name.length() > 16) {
            throw new IllegalArgumentException("The team name '" + name + "' is too long!");
        } else {
            ScorePlayerTeam scoreplayerteam = this.getTeam(name);

            if (scoreplayerteam != null) {
                throw new IllegalArgumentException("A team with the name '" + name + "' already exists!");
            } else {
                scoreplayerteam = new ScorePlayerTeam(this, name);
                this.teams.put(name, scoreplayerteam);
                this.broadcastTeamCreated(scoreplayerteam);
                return scoreplayerteam;
            }
        }
    }

    public void removeTeam(ScorePlayerTeam playerTeam) {
        this.teams.remove(playerTeam.getRegisteredName());

        for (String s : playerTeam.getMembershipCollection()) {
            this.teamMemberships.remove(s);
        }

        this.broadcastTeamRemove(playerTeam);
    }

    public boolean addPlayerToTeam(String player, String newTeam) {
        if (player.length() > 40) {
            throw new IllegalArgumentException("The player name '" + player + "' is too long!");
        } else if (!this.teams.containsKey(newTeam)) {
            return false;
        } else {
            ScorePlayerTeam scoreplayerteam = this.getTeam(newTeam);

            if (this.getPlayersTeam(player) != null) {
                this.removePlayerFromTeams(player);
            }

            this.teamMemberships.put(player, scoreplayerteam);
            scoreplayerteam.getMembershipCollection().add(player);
            return true;
        }
    }

    public boolean removePlayerFromTeams(String playerName) {
        ScorePlayerTeam scoreplayerteam = this.getPlayersTeam(playerName);

        if (scoreplayerteam != null) {
            this.removePlayerFromTeam(playerName, scoreplayerteam);
            return true;
        } else {
            return false;
        }
    }

    public void removePlayerFromTeam(String username, ScorePlayerTeam playerTeam) {
        if (this.getPlayersTeam(username) != playerTeam) {
            throw new IllegalStateException("Player is either on another team or not on any team. Cannot remove from team '" + playerTeam.getRegisteredName() + "'.");
        } else {
            this.teamMemberships.remove(username);
            playerTeam.getMembershipCollection().remove(username);
        }
    }

    public Collection<String> getTeamNames() {
        return this.teams.keySet();
    }

    public Collection<ScorePlayerTeam> getTeams() {
        return this.teams.values();
    }

    public ScorePlayerTeam getPlayersTeam(String username) {
        return this.teamMemberships.get(username);
    }

    public void onScoreObjectiveAdded(ScoreObjective scoreObjectiveIn) {
    }

    public void onObjectiveDisplayNameChanged(ScoreObjective objective) {
    }

    public void onScoreObjectiveRemoved(ScoreObjective objective) {
    }

    public void onScoreUpdated(Score scoreIn) {
    }

    public void broadcastScoreUpdate(String scoreName) {
    }

    public void broadcastScoreUpdate(String scoreName, ScoreObjective objective) {
    }

    public void broadcastTeamCreated(ScorePlayerTeam playerTeam) {
    }

    public void sendTeamUpdate(ScorePlayerTeam playerTeam) {
    }

    public void broadcastTeamRemove(ScorePlayerTeam playerTeam) {
    }

    public void removeEntity(Entity entityIn) {
        if (entityIn != null && !(entityIn instanceof EntityPlayer) && !entityIn.isEntityAlive()) {
            String s = entityIn.getUniqueID().toString();
            this.removeObjectiveFromEntity(s, null);
            this.removePlayerFromTeams(s);
        }
    }
}
