package net.minecraft.profiler;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import net.minecraft.client.option.options.videosettings.PerformanceOption;
import it.unimi.dsi.fastutil.longs.LongArrayList;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.src.Config;
import net.optifine.Lagometer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.Collections;
import java.util.List;
import java.util.Map;

public class Profiler {
    private static final Logger logger = LogManager.getLogger();
    private static final String SCHEDULED_EXECUTABLES = "scheduledExecutables";
    private static final String TICK = "tick";
    private static final String PRE_RENDER_ERRORS = "preRenderErrors";
    private static final String RENDER = "render";
    private static final String DISPLAY = "display";
    private static final int HASH_SCHEDULED_EXECUTABLES = "scheduledExecutables".hashCode();
    private static final int HASH_TICK = "tick".hashCode();
    private static final int HASH_PRE_RENDER_ERRORS = "preRenderErrors".hashCode();
    private static final int HASH_RENDER = "render".hashCode();
    private static final int HASH_DISPLAY = "display".hashCode();
    public final boolean profilerGlobalEnabled = true;
    private final List<String> sectionList = Lists.newArrayList();
    private final LongArrayList timestampList = new LongArrayList();
    private final Map<String, Long> profilingMap = Maps.newHashMap();
    public boolean profilingEnabled;
    private String profilingSection = "";
    private boolean profilerLocalEnabled;

    public Profiler() {
        this.profilerLocalEnabled = this.profilerGlobalEnabled;
    }

    public void clearProfiling() {
        this.profilingMap.clear();
        this.profilingSection = "";
        this.sectionList.clear();
        this.profilerLocalEnabled = this.profilerGlobalEnabled;
    }

    public void startSection(String name) {
        if (Lagometer.isActive()) {
            int i = name.hashCode();

            if (i == HASH_SCHEDULED_EXECUTABLES && name.equals("scheduledExecutables")) {
                Lagometer.timerScheduledExecutables.start();
            } else if (i == HASH_TICK && name.equals("tick") && Config.isMinecraftThread()) {
                Lagometer.timerScheduledExecutables.end();
                Lagometer.timerTick.start();
            } else if (i == HASH_PRE_RENDER_ERRORS && name.equals("preRenderErrors")) {
                Lagometer.timerTick.end();
            }
        }

        if (PerformanceOption.isFastRender()) {
            int j = name.hashCode();

            if (j == HASH_RENDER && name.equals("render")) {
                GlStateManager.clearEnabled = false;
            } else if (j == HASH_DISPLAY && name.equals("display")) {
                GlStateManager.clearEnabled = true;
            }
        }

        if (this.profilerLocalEnabled) {
            if (this.profilingEnabled) {
                if (!this.profilingSection.isEmpty()) {
                    this.profilingSection = this.profilingSection + ".";
                }

                this.profilingSection = this.profilingSection + name;
                this.sectionList.add(this.profilingSection);
                this.timestampList.add(System.nanoTime());
            }
        }
    }

    public void endSection() {
        if (this.profilerLocalEnabled) {
            if (this.profilingEnabled) {
                long i = System.nanoTime();
                long j = this.timestampList.remove(this.timestampList.size() - 1);
                this.sectionList.remove(this.sectionList.size() - 1);
                long k = i - j;

                if (this.profilingMap.containsKey(this.profilingSection)) {
                    this.profilingMap.put(this.profilingSection, this.profilingMap.get(this.profilingSection) + k);
                } else {
                    this.profilingMap.put(this.profilingSection, k);
                }

                if (k > 100000000L) {
                    logger.warn("Something's taking too long! '{}' took aprox {} ms", this.profilingSection, (double) k / 1000000.0D);
                }

                this.profilingSection = !this.sectionList.isEmpty() ? this.sectionList.get(this.sectionList.size() - 1) : "";
            }
        }
    }

    public List<Profiler.Result> getProfilingData(String profilerName) {
        if (!this.profilingEnabled) {
            return null;
        } else {
            long i = this.profilingMap.getOrDefault("root", 0L);
            long j = this.profilingMap.getOrDefault(profilerName, -1L);
            List<Profiler.Result> list = Lists.newArrayList();

            if (!profilerName.isEmpty()) {
                profilerName = profilerName + ".";
            }

            long k = 0L;

            for (String s : this.profilingMap.keySet()) {
                if (s.length() > profilerName.length() && s.startsWith(profilerName) && s.indexOf(".", profilerName.length() + 1) < 0) {
                    k += this.profilingMap.get(s);
                }
            }

            float f = (float) k;

            if (k < j) {
                k = j;
            }

            if (i < k) {
                i = k;
            }

            for (String s1 : this.profilingMap.keySet()) {
                if (s1.length() > profilerName.length() && s1.startsWith(profilerName) && s1.indexOf(".", profilerName.length() + 1) < 0) {
                    long l = this.profilingMap.get(s1);
                    double d0 = (double) l * 100.0D / (double) k;
                    double d1 = (double) l * 100.0D / (double) i;
                    String s2 = s1.substring(profilerName.length());
                    list.add(new Profiler.Result(s2, d0, d1));
                }
            }

            this.profilingMap.replaceAll((s, v) -> this.profilingMap.get(s) * 950L / 1000L);

            if ((float) k > f) {
                list.add(new Profiler.Result("unspecified", (double) ((float) k - f) * 100.0D / (double) k, (double) ((float) k - f) * 100.0D / (double) i));
            }

            Collections.sort(list);
            list.add(0, new Profiler.Result(profilerName, 100.0D, (double) k * 100.0D / (double) i));
            return list;
        }
    }

    public void endStartSection(String name) {
        if (this.profilerLocalEnabled) {
            this.endSection();
            this.startSection(name);
        }
    }

    public String getNameOfLastSection() {
        return this.sectionList.isEmpty() ? "[UNKNOWN]" : this.sectionList.get(this.sectionList.size() - 1);
    }

    public void startSection(Class<?> p_startSection_1_) {
        if (this.profilingEnabled) {
            this.startSection(p_startSection_1_.getSimpleName());
        }
    }

    public static final class Result implements Comparable<Profiler.Result> {
        public final double usePercentage;
        public final double totalUsePercentage;
        public final String profilerName;

        public Result(String profilerName, double usePercentage, double totalUsePercentage) {
            this.profilerName = profilerName;
            this.usePercentage = usePercentage;
            this.totalUsePercentage = totalUsePercentage;
        }

        public int compareTo(Profiler.Result p_compareTo_1_) {
            return p_compareTo_1_.usePercentage < this.usePercentage ? -1 : (p_compareTo_1_.usePercentage > this.usePercentage ? 1 : p_compareTo_1_.profilerName.compareTo(this.profilerName));
        }

        public int getColor() {
            return (this.profilerName.hashCode() & 11184810) + 4473924;
        }
    }
}
