package net.minecraft.potion;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import net.minecraft.client.option.options.videosettings.QualityOption;
import net.minecraft.util.IntegerCache;
import net.optifine.CustomColors;

import java.util.Collection;
import java.util.List;
import java.util.Map;

public class PotionHelper {
    public static final String unusedString = null;
    public static final String sugarEffect = "-0+1-2-3&4-4+13";
    public static final String ghastTearEffect = "+0-1-2-3&4-4+13";
    public static final String spiderEyeEffect = "-0-1+2-3&4-4+13";
    public static final String fermentedSpiderEyeEffect = "-0+3-4+13";
    public static final String speckledMelonEffect = "+0-1+2-3&4-4+13";
    public static final String blazePowderEffect = "+0-1-2+3&4-4+13";
    public static final String magmaCreamEffect = "+0+1-2-3&4-4+13";
    public static final String redstoneEffect = "-5+6-7";
    public static final String glowstoneEffect = "+5-6-7";
    public static final String gunpowderEffect = "+14&13-13";
    public static final String goldenCarrotEffect = "-0+1+2-3+13&4-4";
    public static final String pufferfishEffect = "+0-1+2+3+13&4-4";
    public static final String rabbitFootEffect = "+0+1-2+3&4-4+13";
    private static final Map<Integer, String> potionRequirements = Maps.newHashMap();
    private static final Map<Integer, String> potionAmplifiers = Maps.newHashMap();
    private static final Map<Integer, Integer> DATAVALUE_COLORS = Maps.newHashMap();
    private static final String[] potionPrefixes = new String[]{"potion.prefix.mundane", "potion.prefix.uninteresting", "potion.prefix.bland", "potion.prefix.clear", "potion.prefix.milky", "potion.prefix.diffuse", "potion.prefix.artless", "potion.prefix.thin", "potion.prefix.awkward", "potion.prefix.flat", "potion.prefix.bulky", "potion.prefix.bungling", "potion.prefix.buttered", "potion.prefix.smooth", "potion.prefix.suave", "potion.prefix.debonair", "potion.prefix.thick", "potion.prefix.elegant", "potion.prefix.fancy", "potion.prefix.charming", "potion.prefix.dashing", "potion.prefix.refined", "potion.prefix.cordial", "potion.prefix.sparkling", "potion.prefix.potent", "potion.prefix.foul", "potion.prefix.odorless", "potion.prefix.rank", "potion.prefix.harsh", "potion.prefix.acrid", "potion.prefix.gross", "potion.prefix.stinky"};

    static {
        potionRequirements.put(Potion.regeneration.getId(), "0 & !1 & !2 & !3 & 0+6");
        potionRequirements.put(Potion.moveSpeed.getId(), "!0 & 1 & !2 & !3 & 1+6");
        potionRequirements.put(Potion.fireResistance.getId(), "0 & 1 & !2 & !3 & 0+6");
        potionRequirements.put(Potion.heal.getId(), "0 & !1 & 2 & !3");
        potionRequirements.put(Potion.poison.getId(), "!0 & !1 & 2 & !3 & 2+6");
        potionRequirements.put(Potion.weakness.getId(), "!0 & !1 & !2 & 3 & 3+6");
        potionRequirements.put(Potion.harm.getId(), "!0 & !1 & 2 & 3");
        potionRequirements.put(Potion.moveSlowdown.getId(), "!0 & 1 & !2 & 3 & 3+6");
        potionRequirements.put(Potion.damageBoost.getId(), "0 & !1 & !2 & 3 & 3+6");
        potionRequirements.put(Potion.nightVision.getId(), "!0 & 1 & 2 & !3 & 2+6");
        potionRequirements.put(Potion.invisibility.getId(), "!0 & 1 & 2 & 3 & 2+6");
        potionRequirements.put(Potion.waterBreathing.getId(), "0 & !1 & 2 & 3 & 2+6");
        potionRequirements.put(Potion.jump.getId(), "0 & 1 & !2 & 3 & 3+6");
        potionAmplifiers.put(Potion.moveSpeed.getId(), "5");
        potionAmplifiers.put(Potion.digSpeed.getId(), "5");
        potionAmplifiers.put(Potion.damageBoost.getId(), "5");
        potionAmplifiers.put(Potion.regeneration.getId(), "5");
        potionAmplifiers.put(Potion.harm.getId(), "5");
        potionAmplifiers.put(Potion.heal.getId(), "5");
        potionAmplifiers.put(Potion.resistance.getId(), "5");
        potionAmplifiers.put(Potion.poison.getId(), "5");
        potionAmplifiers.put(Potion.jump.getId(), "5");
    }

    public static boolean checkFlag(int flag, int index) {
        return (flag & 1 << index) != 0;
    }

    private static int isFlagSet(int flag, int index) {
        return checkFlag(flag, index) ? 1 : 0;
    }

    private static int isFlagUnset(int flag, int index) {
        return checkFlag(flag, index) ? 0 : 1;
    }

    public static int getPotionPrefixIndex(int dataValue) {
        return getPotionPrefixIndexFlags(dataValue, 5, 4, 3, 2, 1);
    }

    public static int calcPotionLiquidColor(Collection<PotionEffect> potionEffects) {
        int i = 3694022;

        if (potionEffects != null && !potionEffects.isEmpty()) {
            float f = 0.0F;
            float f1 = 0.0F;
            float f2 = 0.0F;
            float f3 = 0.0F;

            for (PotionEffect potioneffect : potionEffects) {
                if (potioneffect.getIsShowParticles()) {
                    int j = Potion.potionTypes[potioneffect.getPotionID()].getLiquidColor();

                    if (QualityOption.isCustomColors()) {
                        j = CustomColors.getPotionColor(potioneffect.getPotionID(), j);
                    }

                    for (int k = 0; k <= potioneffect.getAmplifier(); ++k) {
                        f += (float) (j >> 16 & 255) / 255.0F;
                        f1 += (float) (j >> 8 & 255) / 255.0F;
                        f2 += (float) (j & 255) / 255.0F;
                        ++f3;
                    }
                }
            }

            if (f3 == 0.0F) {
                return 0;
            } else {
                f = f / f3 * 255.0F;
                f1 = f1 / f3 * 255.0F;
                f2 = f2 / f3 * 255.0F;
                return (int) f << 16 | (int) f1 << 8 | (int) f2;
            }
        } else {
            return QualityOption.isCustomColors() ? CustomColors.getPotionColor(0, i) : i;
        }
    }

    public static boolean getAreAmbient(Collection<PotionEffect> potionEffects) {
        for (PotionEffect potioneffect : potionEffects) {
            if (!potioneffect.getIsAmbient()) {
                return false;
            }
        }

        return true;
    }

    public static int getLiquidColor(int dataValue, boolean bypassCache) {
        Integer integer = IntegerCache.getInteger(dataValue);

        if (!bypassCache) {
            if (DATAVALUE_COLORS.containsKey(integer)) {
                return DATAVALUE_COLORS.get(integer);
            } else {
                int i = calcPotionLiquidColor(getPotionEffects(integer, false));
                DATAVALUE_COLORS.put(integer, i);
                return i;
            }
        } else {
            return calcPotionLiquidColor(getPotionEffects(integer, true));
        }
    }

    public static String getPotionPrefix(int dataValue) {
        int i = getPotionPrefixIndex(dataValue);
        return potionPrefixes[i];
    }

    private static int getPotionEffect(boolean flagUnset, boolean amplifierDefined, boolean negativeEffect, int type, int index, int amplifier, int flagMask) {
        int i = 0;

        if (flagUnset) {
            i = isFlagUnset(flagMask, index);
        } else if (type != -1) {
            if (type == 0 && countSetFlags(flagMask) == index) {
                i = 1;
            } else if (type == 1 && countSetFlags(flagMask) > index) {
                i = 1;
            } else if (type == 2 && countSetFlags(flagMask) < index) {
                i = 1;
            }
        } else {
            i = isFlagSet(flagMask, index);
        }

        if (amplifierDefined) {
            i *= amplifier;
        }

        if (negativeEffect) {
            i *= -1;
        }

        return i;
    }

    private static int countSetFlags(int flagMask) {
        int i;

        for (i = 0; flagMask > 0; ++i) {
            flagMask &= flagMask - 1;
        }

        return i;
    }

    private static int parsePotionEffects(String effectText, int start, int end, int flagMask) {
        if (start < effectText.length() && end >= 0 && start < end) {
            int i = effectText.indexOf(124, start);

            if (i >= 0 && i < end) {
                int l1 = parsePotionEffects(effectText, start, i - 1, flagMask);

                if (l1 > 0) {
                    return l1;
                } else {
                    int j2 = parsePotionEffects(effectText, i + 1, end, flagMask);
                    return Math.max(j2, 0);
                }
            } else {
                int j = effectText.indexOf(38, start);

                if (j >= 0 && j < end) {
                    int i2 = parsePotionEffects(effectText, start, j - 1, flagMask);

                    if (i2 <= 0) {
                        return 0;
                    } else {
                        int k2 = parsePotionEffects(effectText, j + 1, end, flagMask);
                        return k2 <= 0 ? 0 : (Math.max(i2, k2));
                    }
                } else {
                    boolean flag = false;
                    boolean flag1 = false;
                    boolean flag2 = false;
                    boolean flag3 = false;
                    boolean flag4 = false;
                    int k = -1;
                    int l = 0;
                    int i1 = 0;
                    int j1 = 0;

                    for (int k1 = start; k1 < end; ++k1) {
                        char c0 = effectText.charAt(k1);

                        if (c0 >= 48 && c0 <= 57) {
                            if (flag) {
                                i1 = c0 - 48;
                                flag1 = true;
                            } else {
                                l = l * 10;
                                l = l + (c0 - 48);
                                flag2 = true;
                            }
                        } else if (c0 == 42) {
                            flag = true;
                        } else if (c0 == 33) {
                            if (flag2) {
                                j1 += getPotionEffect(flag3, flag1, flag4, k, l, i1, flagMask);
                                flag4 = false;
                                flag = false;
                                flag1 = false;
                                flag2 = false;
                                i1 = 0;
                                l = 0;
                                k = -1;
                            }

                            flag3 = true;
                        } else if (c0 == 45) {
                            if (flag2) {
                                j1 += getPotionEffect(flag3, flag1, flag4, k, l, i1, flagMask);
                                flag3 = false;
                                flag = false;
                                flag1 = false;
                                flag2 = false;
                                i1 = 0;
                                l = 0;
                                k = -1;
                            }

                            flag4 = true;
                        } else if (c0 != 61 && c0 != 60 && c0 != 62) {
                            if (c0 == 43 && flag2) {
                                j1 += getPotionEffect(flag3, flag1, flag4, k, l, i1, flagMask);
                                flag3 = false;
                                flag4 = false;
                                flag = false;
                                flag1 = false;
                                flag2 = false;
                                i1 = 0;
                                l = 0;
                                k = -1;
                            }
                        } else {
                            if (flag2) {
                                j1 += getPotionEffect(flag3, flag1, flag4, k, l, i1, flagMask);
                                flag3 = false;
                                flag4 = false;
                                flag = false;
                                flag1 = false;
                                flag2 = false;
                                i1 = 0;
                                l = 0;
                            }

                            if (c0 == 61) {
                                k = 0;
                            } else if (c0 == 60) {
                                k = 2;
                            } else {
                                k = 1;
                            }
                        }
                    }

                    if (flag2) {
                        j1 += getPotionEffect(flag3, flag1, flag4, k, l, i1, flagMask);
                    }

                    return j1;
                }
            }
        } else {
            return 0;
        }
    }

    public static List<PotionEffect> getPotionEffects(int flagMask, boolean usable) {
        List<PotionEffect> list = null;

        for (Potion potion : Potion.potionTypes) {
            if (potion != null && (!potion.isUsable() || usable)) {
                String s = potionRequirements.get(potion.getId());

                if (s != null) {
                    int i = parsePotionEffects(s, 0, s.length(), flagMask);

                    if (i > 0) {
                        int j = 0;
                        String s1 = potionAmplifiers.get(potion.getId());

                        if (s1 != null) {
                            j = parsePotionEffects(s1, 0, s1.length(), flagMask);

                            if (j < 0) {
                                j = 0;
                            }
                        }

                        if (potion.isInstant()) {
                            i = 1;
                        } else {
                            i = 1200 * (i * 3 + (i - 1) * 2);
                            i = i >> j;
                            i = (int) Math.round((double) i * potion.getEffectiveness());

                            if ((flagMask & 16384) != 0) {
                                i = (int) Math.round((double) i * 0.75D + 0.5D);
                            }
                        }

                        if (list == null) {
                            list = Lists.newArrayList();
                        }

                        PotionEffect potioneffect = new PotionEffect(potion.getId(), i, j);

                        if ((flagMask & 16384) != 0) {
                            potioneffect.setSplashPotion(true);
                        }

                        list.add(potioneffect);
                    }
                }
            }
        }

        return list;
    }

    private static int brewBitOperations(int flagMask, int index, boolean shouldCheck, boolean toggle, boolean unset) {
        if (unset) {
            if (!checkFlag(flagMask, index)) {
                return 0;
            }
        } else if (shouldCheck) {
            flagMask &= ~(1 << index);
        } else if (toggle) {
            if ((flagMask & 1 << index) == 0) {
                flagMask |= 1 << index;
            } else {
                flagMask &= ~(1 << index);
            }
        } else {
            flagMask |= 1 << index;
        }

        return flagMask;
    }

    public static int applyIngredient(int flagMask, String ingredient) {
        int i = 0;
        int j = ingredient.length();
        boolean flag = false;
        boolean flag1 = false;
        boolean flag2 = false;
        boolean flag3 = false;
        int k = 0;

        for (int l = i; l < j; ++l) {
            char c0 = ingredient.charAt(l);

            if (c0 >= 48 && c0 <= 57) {
                k = k * 10;
                k = k + (c0 - 48);
                flag = true;
            } else if (c0 == 33) {
                if (flag) {
                    flagMask = brewBitOperations(flagMask, k, flag2, flag1, flag3);
                    flag3 = false;
                    flag2 = false;
                    flag = false;
                    k = 0;
                }

                flag1 = true;
            } else if (c0 == 45) {
                if (flag) {
                    flagMask = brewBitOperations(flagMask, k, flag2, flag1, flag3);
                    flag3 = false;
                    flag1 = false;
                    flag = false;
                    k = 0;
                }

                flag2 = true;
            } else if (c0 == 43) {
                if (flag) {
                    flagMask = brewBitOperations(flagMask, k, flag2, flag1, flag3);
                    flag3 = false;
                    flag1 = false;
                    flag2 = false;
                    flag = false;
                    k = 0;
                }
            } else if (c0 == 38) {
                if (flag) {
                    flagMask = brewBitOperations(flagMask, k, flag2, flag1, flag3);
                    flag1 = false;
                    flag2 = false;
                    flag = false;
                    k = 0;
                }

                flag3 = true;
            }
        }

        if (flag) {
            flagMask = brewBitOperations(flagMask, k, flag2, flag1, flag3);
        }

        return flagMask & 32767;
    }

    public static int getPotionPrefixIndexFlags(int flagMask, int flag1, int flag2, int flag3, int flag4, int flag5) {
        return (checkFlag(flagMask, flag1) ? 16 : 0) | (checkFlag(flagMask, flag2) ? 8 : 0) | (checkFlag(flagMask, flag3) ? 4 : 0) | (checkFlag(flagMask, flag4) ? 2 : 0) | (checkFlag(flagMask, flag5) ? 1 : 0);
    }
}
