package net.minecraft.network.play.server;

import com.google.common.collect.Lists;
import net.minecraft.network.Packet;
import net.minecraft.network.PacketBuffer;
import net.minecraft.network.play.INetHandlerPlayClient;
import net.minecraft.world.chunk.Chunk;
import net.minecraft.world.chunk.storage.ExtendedBlockStorage;

import java.util.List;

public class S21PacketChunkData extends Packet<INetHandlerPlayClient> {
    private int chunkX;
    private int chunkZ;
    private S21PacketChunkData.Extracted extractedData;
    private boolean loadChunk;

    public S21PacketChunkData() {
    }

    public S21PacketChunkData(Chunk chunkIn, boolean loadChunk, int mask) {
        this.chunkX = chunkIn.xPosition;
        this.chunkZ = chunkIn.zPosition;
        this.loadChunk = loadChunk;
        this.extractedData = getExtractedData(chunkIn, loadChunk, !chunkIn.getWorld().provider.getHasNoSky(), mask);
    }

    protected static int calculateByteChunk(int section, boolean skyLight, boolean biomes) {
        int i = section * 2 * 16 * 16 * 16;
        int j = section * 16 * 16 * 16 / 2;
        int k = skyLight ? section * 16 * 16 * 16 / 2 : 0;
        int l = biomes ? 256 : 0;
        return i + j + k + l;
    }

    public static S21PacketChunkData.Extracted getExtractedData(Chunk chunk, boolean biomes, boolean skylight, int mask) {
        ExtendedBlockStorage[] aextendedblockstorage = chunk.getBlockStorageArray();
        S21PacketChunkData.Extracted s21packetchunkdata$extracted = new S21PacketChunkData.Extracted();
        List<ExtendedBlockStorage> list = Lists.newArrayList();

        for (int i = 0; i < aextendedblockstorage.length; ++i) {
            ExtendedBlockStorage extendedblockstorage = aextendedblockstorage[i];

            if (extendedblockstorage != null && (!biomes || !extendedblockstorage.isEmpty()) && (mask & 1 << i) != 0) {
                s21packetchunkdata$extracted.dataSize |= 1 << i;
                list.add(extendedblockstorage);
            }
        }

        s21packetchunkdata$extracted.data = new byte[calculateByteChunk(Integer.bitCount(s21packetchunkdata$extracted.dataSize), skylight, biomes)];
        int j = 0;

        for (ExtendedBlockStorage extendedblockstorage1 : list) {
            char[] achar = extendedblockstorage1.getData();

            for (char c0 : achar) {
                s21packetchunkdata$extracted.data[j++] = (byte) (c0 & 255);
                s21packetchunkdata$extracted.data[j++] = (byte) (c0 >> 8 & 255);
            }
        }

        for (ExtendedBlockStorage extendedblockstorage2 : list) {
            j = copyArray(extendedblockstorage2.getBlocklightArray().getData(), s21packetchunkdata$extracted.data, j);
        }

        if (skylight) {
            for (ExtendedBlockStorage extendedblockstorage3 : list) {
                j = copyArray(extendedblockstorage3.getSkylightArray().getData(), s21packetchunkdata$extracted.data, j);
            }
        }

        if (biomes) {
            copyArray(chunk.getBiomeArray(), s21packetchunkdata$extracted.data, j);
        }

        return s21packetchunkdata$extracted;
    }

    private static int copyArray(byte[] source, byte[] destination, int length) {
        System.arraycopy(source, 0, destination, length, source.length);
        return length + source.length;
    }

    public void readPacketData(PacketBuffer buf) {
        this.chunkX = buf.readInt();
        this.chunkZ = buf.readInt();
        this.loadChunk = buf.readBoolean();
        this.extractedData = new S21PacketChunkData.Extracted();
        this.extractedData.dataSize = buf.readShort();
        this.extractedData.data = buf.readByteArray();
    }

    public void writePacketData(PacketBuffer buf) {
        buf.writeInt(this.chunkX);
        buf.writeInt(this.chunkZ);
        buf.writeBoolean(this.loadChunk);
        buf.writeShort((short) (this.extractedData.dataSize & 65535));
        buf.writeByteArray(this.extractedData.data);
    }

    public void processPacket(INetHandlerPlayClient handler) {
        handler.handleChunkData(this);
    }

    public byte[] getExtractedDataBytes() {
        return this.extractedData.data;
    }

    public int getChunkX() {
        return this.chunkX;
    }

    public int getChunkZ() {
        return this.chunkZ;
    }

    public int getExtractedSize() {
        return this.extractedData.dataSize;
    }

    public boolean doChunkLoad() {
        return this.loadChunk;
    }

    public static class Extracted {
        public byte[] data;
        public int dataSize;
    }
}
