package net.minecraft.network.play.server;

import com.google.common.collect.Lists;
import net.minecraft.entity.ai.attributes.AttributeModifier;
import net.minecraft.entity.ai.attributes.IAttributeInstance;
import net.minecraft.network.Packet;
import net.minecraft.network.PacketBuffer;
import net.minecraft.network.play.INetHandlerPlayClient;

import java.util.Collection;
import java.util.List;
import java.util.UUID;

public class S20PacketEntityProperties extends Packet<INetHandlerPlayClient> {
    private final List<S20PacketEntityProperties.Snapshot> snapshots = Lists.newArrayList();
    private int entityId;

    public S20PacketEntityProperties() {
    }

    public S20PacketEntityProperties(int entityIdIn, Collection<IAttributeInstance> attributes) {
        this.entityId = entityIdIn;

        for (IAttributeInstance iattributeinstance : attributes) {
            this.snapshots.add(new S20PacketEntityProperties.Snapshot(iattributeinstance.getAttribute().getAttributeUnlocalizedName(), iattributeinstance.getBaseValue(), iattributeinstance.getModifiers()));
        }
    }

    public void readPacketData(PacketBuffer buf) {
        this.entityId = buf.readVarIntFromBuffer();
        int i = buf.readInt();

        for (int j = 0; j < i; ++j) {
            String s = buf.readStringFromBuffer(64);
            double d0 = buf.readDouble();
            List<AttributeModifier> list = Lists.newArrayList();
            int k = buf.readVarIntFromBuffer();

            for (int l = 0; l < k; ++l) {
                UUID uuid = buf.readUuid();
                list.add(new AttributeModifier(uuid, "Unknown synced attribute modifier", buf.readDouble(), buf.readByte()));
            }

            this.snapshots.add(new S20PacketEntityProperties.Snapshot(s, d0, list));
        }
    }

    public void writePacketData(PacketBuffer buf) {
        buf.writeVarIntToBuffer(this.entityId);
        buf.writeInt(this.snapshots.size());

        for (S20PacketEntityProperties.Snapshot s20packetentityproperties$snapshot : this.snapshots) {
            buf.writeString(s20packetentityproperties$snapshot.getName());
            buf.writeDouble(s20packetentityproperties$snapshot.getBaseValue());
            buf.writeVarIntToBuffer(s20packetentityproperties$snapshot.getModifiers().size());

            for (AttributeModifier attributemodifier : s20packetentityproperties$snapshot.getModifiers()) {
                buf.writeUuid(attributemodifier.getID());
                buf.writeDouble(attributemodifier.getAmount());
                buf.writeByte(attributemodifier.getOperation());
            }
        }
    }

    public void processPacket(INetHandlerPlayClient handler) {
        handler.handleEntityProperties(this);
    }

    public int getEntityId() {
        return this.entityId;
    }

    public List<S20PacketEntityProperties.Snapshot> getSnapshots() {
        return this.snapshots;
    }

    static public class Snapshot {
        private final String name;
        private final double baseValue;
        private final Collection<AttributeModifier> modifiers;

        public Snapshot(String name, double baseValue, Collection<AttributeModifier> modifiers) {
            this.name = name;
            this.baseValue = baseValue;
            this.modifiers = modifiers;
        }

        public String getName() {
            return this.name;
        }

        public double getBaseValue() {
            return this.baseValue;
        }

        public Collection<AttributeModifier> getModifiers() {
            return this.modifiers;
        }
    }
}
