package net.minecraft.nbt;

import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonParser;
import com.mojang.authlib.GameProfile;
import com.mojang.authlib.properties.Property;
import net.minecraft.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Objects;
import java.util.UUID;

public final class NBTUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(NBTUtil.class);

    /**
     * Fix skull crash on some servers
     * @author Izikie
     * @from Izikie/Radiant
     */
    private static String fixSkullCrash(NBTTagCompound nbtTagCompound) {
        try {
            String valueURL = nbtTagCompound.getString("Value");
            String decodedValueString = new String(Base64.getDecoder().decode(valueURL), StandardCharsets.UTF_8);
            JsonObject jsonObject = JsonParser.parseString(decodedValueString).getAsJsonObject();

            String url = jsonObject
                .getAsJsonObject("textures")
                .getAsJsonObject("SKIN")
                .get("url")
                .getAsString();

            if (url.matches("(http|https)://textures.minecraft.net/texture/.*")) {
                return valueURL;
            } else {
                jsonObject.remove("textures");
                return Base64.getEncoder().encodeToString(jsonObject.toString().getBytes(StandardCharsets.UTF_8));
            }
        } catch (IllegalArgumentException | IllegalStateException e) {
            LOGGER.error("Failed to decode or parse skull value: {}", e.getMessage());
            return "";
        } catch (JsonParseException exception) {
            LOGGER.error("Could not parse skull owner data: {}", exception.getMessage());
            return "";
        }
    }

    public static GameProfile readGameProfileFromNBT(NBTTagCompound compound) {
        String s = null;
        String s1 = null;

        if (compound.hasKey("Name", 8))
            s = compound.getString("Name");

        if (compound.hasKey("Id", 8))
            s1 = compound.getString("Id");

        if (StringUtils.isNullOrEmpty(s) && StringUtils.isNullOrEmpty(s1))
            return null;

        UUID uuid;

        try {
            uuid = UUID.fromString(Objects.requireNonNull(s1));
        } catch (Throwable var12) {
            uuid = null;
        }

        GameProfile gameprofile = new GameProfile(uuid, s);

        if (compound.hasKey("Properties", 10)) {
            NBTTagCompound propertiesTag = compound.getCompoundTag("Properties");

            for (String s2 : propertiesTag.getKeySet()) {
                NBTTagList tagList = propertiesTag.getTagList(s2, 10);

                for (int i = 0; i < tagList.tagCount(); ++i) {
                    NBTTagCompound tagCompound = tagList.getCompoundTagAt(i);
                    String s3 = fixSkullCrash(tagCompound);

                    if (tagCompound.hasKey("Signature", 8)) {
                        gameprofile.getProperties().put(s2, new Property(s2, s3, tagCompound.getString("Signature")));
                    } else {
                        gameprofile.getProperties().put(s2, new Property(s2, s3));
                    }
                }
            }
        }

        return gameprofile;
    }

    public static NBTTagCompound writeGameProfile(NBTTagCompound tagCompound, GameProfile profile) {
        if (!StringUtils.isNullOrEmpty(profile.getName())) {
            tagCompound.setString("Name", profile.getName());
        }

        if (profile.getId() != null) {
            tagCompound.setString("Id", profile.getId().toString());
        }

        if (!profile.getProperties().isEmpty()) {
            NBTTagCompound nbttagcompound = new NBTTagCompound();

            for (String s : profile.getProperties().keySet()) {
                NBTTagList nbttaglist = new NBTTagList();

                for (Property property : profile.getProperties().get(s)) {
                    NBTTagCompound nbttagcompound1 = new NBTTagCompound();
                    nbttagcompound1.setString("Value", property.getValue());

                    if (property.hasSignature()) {
                        nbttagcompound1.setString("Signature", property.getSignature());
                    }

                    nbttaglist.appendTag(nbttagcompound1);
                }

                nbttagcompound.setTag(s, nbttaglist);
            }

            tagCompound.setTag("Properties", nbttagcompound);
        }

        return tagCompound;
    }

    public static boolean areNBTEquals(NBTBase nbt1, NBTBase nbt2, boolean compareTagList) {
        if (nbt1 == nbt2) {
            return true;
        } else if (nbt1 == null) {
            return true;
        } else if (nbt2 == null) {
            return false;
        } else if (!nbt1.getClass().equals(nbt2.getClass())) {
            return false;
        } else if (nbt1 instanceof NBTTagCompound nbttagcompound) {
            NBTTagCompound nbttagcompound1 = (NBTTagCompound) nbt2;

            for (String s : nbttagcompound.getKeySet()) {
                NBTBase nbtbase1 = nbttagcompound.getTag(s);

                if (!areNBTEquals(nbtbase1, nbttagcompound1.getTag(s), compareTagList)) {
                    return false;
                }
            }

            return true;
        } else if (nbt1 instanceof NBTTagList nbttaglist && compareTagList) {
            NBTTagList nbttaglist1 = (NBTTagList) nbt2;

            if (nbttaglist.tagCount() == 0) {
                return nbttaglist1.tagCount() == 0;
            } else {
                for (int i = 0; i < nbttaglist.tagCount(); ++i) {
                    NBTBase nbtbase = nbttaglist.get(i);
                    boolean flag = false;

                    for (int j = 0; j < nbttaglist1.tagCount(); ++j) {
                        if (areNBTEquals(nbtbase, nbttaglist1.get(j), true)) {
                            flag = true;
                            break;
                        }
                    }

                    if (!flag) {
                        return false;
                    }
                }

                return true;
            }
        } else {
            return nbt1.equals(nbt2);
        }
    }
}
