package net.minecraft.client.util;

import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.List;

public class JsonException extends IOException {
    private final List<JsonException.Entry> entries = Lists.newArrayList();
    private final String exceptionMessage;

    public JsonException(String message) {
        this.entries.add(new JsonException.Entry());
        this.exceptionMessage = message;
    }

    public JsonException(String message, Throwable cause) {
        super(cause);
        this.entries.add(new JsonException.Entry());
        this.exceptionMessage = message;
    }

    public static JsonException forException(Exception exception) {
        if (exception instanceof JsonException) {
            return (JsonException) exception;
        } else {
            String s = exception.getMessage();

            if (exception instanceof FileNotFoundException) {
                s = "File not found";
            }

            return new JsonException(s, exception);
        }
    }

    public void prependJsonKey(String key) {
        this.entries.get(0).addJsonKey(key);
    }

    public void setFilenameAndFlush(String filenameIn) {
        this.entries.get(0).filename = filenameIn;
        this.entries.add(0, new JsonException.Entry());
    }

    public String getMessage() {
        return "Invalid " + this.entries.get(this.entries.size() - 1).toString() + ": " + this.exceptionMessage;
    }

    public static class Entry {
        private final List<String> jsonKeys;
        private String filename;

        private Entry() {
            this.filename = null;
            this.jsonKeys = Lists.newArrayList();
        }

        private void addJsonKey(String key) {
            this.jsonKeys.add(0, key);
        }

        public String getJsonKeys() {
            return StringUtils.join(this.jsonKeys, "->");
        }

        public String toString() {
            return this.filename != null ? (!this.jsonKeys.isEmpty() ? this.filename + " " + this.getJsonKeys() : this.filename) : (!this.jsonKeys.isEmpty() ? "(Unknown file) " + this.getJsonKeys() : "(Unknown file)");
        }
    }
}
