package net.minecraft.client.resources;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import net.minecraft.client.option.options.general.OtherOption;
import net.minecraft.client.resources.data.IMetadataSerializer;
import net.minecraft.client.resources.data.LanguageMetadataSection;
import net.minecraft.util.StringTranslate;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;

public class LanguageManager implements IResourceManagerReloadListener {
    public static final Locale currentLocale = new Locale();
    private static final Logger logger = LogManager.getLogger();
    private final IMetadataSerializer theMetadataSerializer;
    private final Map<String, Language> languageMap = Maps.newHashMap();

    public LanguageManager(IMetadataSerializer theMetadataSerializerIn) {
        this.theMetadataSerializer = theMetadataSerializerIn;
    }

    public void parseLanguageMetadata(List<IResourcePack> resourcesPacks) {
        this.languageMap.clear();

        for (IResourcePack iresourcepack : resourcesPacks) {
            try {
                LanguageMetadataSection languagemetadatasection = iresourcepack.getPackMetadata(this.theMetadataSerializer, "language");

                if (languagemetadatasection != null) {
                    for (Language language : languagemetadatasection.getLanguages()) {
                        if (!this.languageMap.containsKey(language.getLanguageCode())) {
                            this.languageMap.put(language.getLanguageCode(), language);
                        }
                    }
                }
            } catch (RuntimeException | IOException runtimeexception) {
                logger.warn("Unable to parse metadata section of resourcepack: {}", iresourcepack.getPackName(), runtimeexception);
            }
        }
    }

    public void onResourceManagerReload(IResourceManager resourceManager) throws IOException {
        List<String> list = Lists.newArrayList("en_US");

        String currentLanguage = OtherOption.getLanguage();

        if (!"en_US".equals(currentLanguage))
            list.add(currentLanguage);

        currentLocale.loadLocaleDataFiles(resourceManager, list);
        StringTranslate.replaceWith(currentLocale.properties);
    }

    public boolean isCurrentLocaleUnicode() {
        return currentLocale.isUnicode();
    }

    public boolean isCurrentLanguageBidirectional() {
        return this.getCurrentLanguage() != null && this.getCurrentLanguage().isBidirectional();
    }

    public Language getLanguageByLanguageCode(String code) {
        return this.languageMap.containsKey(code) ? this.languageMap.get(code) : this.languageMap.get("en_US");
    }

    public Language getCurrentLanguage() {
        return this.getLanguageByLanguageCode(OtherOption.getLanguage());
    }

    public SortedSet<String> getLanguageCodes() {
        return Sets.newTreeSet(this.languageMap.keySet());
    }

    public static String format(String translateKey, Object... parameters) {
        return currentLocale.formatMessage(translateKey, parameters);
    }
}
