package net.minecraft.client.renderer.entity;

import com.google.common.collect.Maps;
import net.minecraft.client.Minecraft;
import net.minecraft.client.model.ModelHorse;
import net.minecraft.client.option.options.videosettings.TexturesOption;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.texture.LayeredTexture;
import net.minecraft.entity.Entity;
import net.minecraft.entity.passive.EntityHorse;
import net.minecraft.util.ResourceLocation;
import org.lwjgl.opengl.GL11;

import java.util.Map;

import static net.minecraft.client.MinecraftInstance.mc;

public class RenderHorse extends RenderLiving<EntityHorse> {
    private static final Map<String, ResourceLocation> LAYERED_LOCATION_CACHE = Maps.newHashMap();
    private static final ResourceLocation whiteHorseTextures = new ResourceLocation("textures/entity/horse/horse_white.png");
    private static final ResourceLocation muleTextures = new ResourceLocation("textures/entity/horse/mule.png");
    private static final ResourceLocation donkeyTextures = new ResourceLocation("textures/entity/horse/donkey.png");
    private static final ResourceLocation zombieHorseTextures = new ResourceLocation("textures/entity/horse/horse_zombie.png");
    private static final ResourceLocation skeletonHorseTextures = new ResourceLocation("textures/entity/horse/horse_skeleton.png");

    private boolean shouldMakeTransparent;

    public RenderHorse(RenderManager rendermanagerIn, ModelHorse model, float shadowSizeIn) {
        super(rendermanagerIn, model, shadowSizeIn);
    }

    protected void preRenderCallback(EntityHorse entitylivingbaseIn, float partialTickTime) {
        float f = 1.0F;
        int i = entitylivingbaseIn.getHorseType();

        if (i == 1) {
            f *= 0.87F;
        } else if (i == 2) {
            f *= 0.92F;
        }

        GlStateManager.scale(f, f, f);
        super.preRenderCallback(entitylivingbaseIn, partialTickTime);
    }

    protected ResourceLocation getEntityTexture(EntityHorse entity) {
        if (!entity.hasLayeredTextures()) {
            return switch (entity.getHorseType()) {
                case 1 -> donkeyTextures;
                case 2 -> muleTextures;
                case 3 -> zombieHorseTextures;
                case 4 -> skeletonHorseTextures;
                default -> whiteHorseTextures;
            };
        } else {
            return this.getLayerTexture(entity);
        }
    }

    private ResourceLocation getLayerTexture(EntityHorse horse) {
        String s = horse.getHorseTexture();

        if (!horse.hasTexture()) {
            return null;
        } else {
            ResourceLocation resourcelocation = LAYERED_LOCATION_CACHE.get(s);

            if (resourcelocation == null) {
                resourcelocation = new ResourceLocation(s);
                Minecraft.getMinecraft().getTextureManager().loadTexture(resourcelocation, new LayeredTexture(horse.getVariantTexturePaths()));
                LAYERED_LOCATION_CACHE.put(s, resourcelocation);
            }

            return resourcelocation;
        }
    }

    @Override
    public void doRender(EntityHorse entity, double x, double y, double z, float entityYaw, float partialTicks) {

        float riddenHorseOpacity = TexturesOption.getRiddenHorseOpacity();

        if (riddenHorseOpacity < 1f) {
            Entity ridingEntity = mc.thePlayer.ridingEntity;

            if (ridingEntity != null) {
                shouldMakeTransparent = ridingEntity == entity;
                if (shouldMakeTransparent) {
                    GlStateManager.enableBlend();
                    GlStateManager.enableAlpha();
                    GlStateManager.alphaFunc(GL11.GL_GREATER, 0.1F);
                    GlStateManager.tryBlendFuncSeparate(GL11.GL_SRC_ALPHA, GL11.GL_ONE_MINUS_SRC_ALPHA, GL11.GL_ONE, GL11.GL_ZERO);
                    GlStateManager.color(1, 1, 1, riddenHorseOpacity);
                }
            }
        }

        super.doRender(entity, x, y, z, entityYaw, partialTicks);

        if (shouldMakeTransparent) {
            GlStateManager.disableBlend();
            GlStateManager.disableAlpha();
            GlStateManager.color(1, 1, 1, 1);
        }
    }
}
