package net.minecraft.client.renderer.block.model;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.*;
import net.minecraft.client.resources.model.ModelRotation;
import net.minecraft.util.JsonUtils;
import net.minecraft.util.ResourceLocation;

import java.io.Reader;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

public class ModelBlockDefinition {
    static final Gson GSON = (new GsonBuilder()).registerTypeAdapter(ModelBlockDefinition.class, new ModelBlockDefinition.Deserializer()).registerTypeAdapter(ModelBlockDefinition.Variant.class, new ModelBlockDefinition.Variant.Deserializer()).create();
    private final Map<String, ModelBlockDefinition.Variants> mapVariants = Maps.newHashMap();

    public ModelBlockDefinition(Collection<ModelBlockDefinition.Variants> variantsList) {
        for (ModelBlockDefinition.Variants modelblockdefinition$variants : variantsList) {
            this.mapVariants.put(modelblockdefinition$variants.name, modelblockdefinition$variants);
        }
    }

    public ModelBlockDefinition(List<ModelBlockDefinition> definitions) {
        for (ModelBlockDefinition modelblockdefinition : definitions) {
            this.mapVariants.putAll(modelblockdefinition.mapVariants);
        }
    }

    public static ModelBlockDefinition parseFromReader(Reader reader) {
        return GSON.fromJson(reader, ModelBlockDefinition.class);
    }

    public ModelBlockDefinition.Variants getVariants(String key) {
        ModelBlockDefinition.Variants modelblockdefinition$variants = this.mapVariants.get(key);

        if (modelblockdefinition$variants == null) {
            throw new MissingVariantException();
        } else {
            return modelblockdefinition$variants;
        }
    }

    public boolean equals(Object p_equals_1_) {
        if (this == p_equals_1_) {
            return true;
        } else if (p_equals_1_ instanceof ModelBlockDefinition modelblockdefinition) {
            return this.mapVariants.equals(modelblockdefinition.mapVariants);
        } else {
            return false;
        }
    }

    public int hashCode() {
        return this.mapVariants.hashCode();
    }

    public static class Deserializer implements JsonDeserializer<ModelBlockDefinition> {
        public ModelBlockDefinition deserialize(JsonElement p_deserialize_1_, Type p_deserialize_2_, JsonDeserializationContext p_deserialize_3_) throws JsonParseException {
            JsonObject jsonobject = p_deserialize_1_.getAsJsonObject();
            List<ModelBlockDefinition.Variants> list = this.parseVariantsList(p_deserialize_3_, jsonobject);
            return new ModelBlockDefinition(list);
        }

        protected List<ModelBlockDefinition.Variants> parseVariantsList(JsonDeserializationContext context, JsonObject jsonObject) {
            JsonObject jsonobject = JsonUtils.getJsonObject(jsonObject, "variants");
            List<ModelBlockDefinition.Variants> list = Lists.newArrayList();

            for (Entry<String, JsonElement> entry : jsonobject.entrySet()) {
                list.add(this.parseVariants(context, entry));
            }

            return list;
        }

        protected ModelBlockDefinition.Variants parseVariants(JsonDeserializationContext context, Entry<String, JsonElement> jsonMap) {
            String s = jsonMap.getKey();
            List<ModelBlockDefinition.Variant> list = Lists.newArrayList();
            JsonElement jsonelement = jsonMap.getValue();

            if (jsonelement.isJsonArray()) {
                for (JsonElement jsonelement1 : jsonelement.getAsJsonArray()) {
                    list.add(context.deserialize(jsonelement1, Variant.class));
                }
            } else {
                list.add(context.deserialize(jsonelement, Variant.class));
            }

            return new ModelBlockDefinition.Variants(s, list);
        }
    }

    public static class MissingVariantException extends RuntimeException {
    }

    public static class Variant {
        private final ResourceLocation modelLocation;
        private final ModelRotation modelRotation;
        private final boolean uvLock;
        private final int weight;

        public Variant(ResourceLocation modelLocationIn, ModelRotation modelRotationIn, boolean uvLockIn, int weightIn) {
            this.modelLocation = modelLocationIn;
            this.modelRotation = modelRotationIn;
            this.uvLock = uvLockIn;
            this.weight = weightIn;
        }

        public ResourceLocation getModelLocation() {
            return this.modelLocation;
        }

        public ModelRotation getRotation() {
            return this.modelRotation;
        }

        public boolean isUvLocked() {
            return this.uvLock;
        }

        public int getWeight() {
            return this.weight;
        }

        public boolean equals(Object p_equals_1_) {
            if (this == p_equals_1_) {
                return true;
            } else if (!(p_equals_1_ instanceof Variant modelblockdefinition$variant)) {
                return false;
            } else {
                return this.modelLocation.equals(modelblockdefinition$variant.modelLocation) && this.modelRotation == modelblockdefinition$variant.modelRotation && this.uvLock == modelblockdefinition$variant.uvLock;
            }
        }

        public int hashCode() {
            int i = this.modelLocation.hashCode();
            i = 31 * i + (this.modelRotation != null ? this.modelRotation.hashCode() : 0);
            i = 31 * i + (this.uvLock ? 1 : 0);
            return i;
        }

        public static class Deserializer implements JsonDeserializer<ModelBlockDefinition.Variant> {
            public ModelBlockDefinition.Variant deserialize(JsonElement p_deserialize_1_, Type p_deserialize_2_, JsonDeserializationContext p_deserialize_3_) throws JsonParseException {
                JsonObject jsonobject = p_deserialize_1_.getAsJsonObject();
                String s = this.parseModel(jsonobject);
                ModelRotation modelrotation = this.parseRotation(jsonobject);
                boolean flag = this.parseUvLock(jsonobject);
                int i = this.parseWeight(jsonobject);
                return new ModelBlockDefinition.Variant(this.makeModelLocation(s), modelrotation, flag, i);
            }

            private ResourceLocation makeModelLocation(String textureLocation) {
                ResourceLocation resourcelocation = new ResourceLocation(textureLocation);
                resourcelocation = new ResourceLocation(resourcelocation.getResourceDomain(), "block/" + resourcelocation.getResourcePath());
                return resourcelocation;
            }

            private boolean parseUvLock(JsonObject jsonObj) {
                return JsonUtils.getBoolean(jsonObj, "uvlock", false);
            }

            protected ModelRotation parseRotation(JsonObject jsonObj) {
                int i = JsonUtils.getInt(jsonObj, "x", 0);
                int j = JsonUtils.getInt(jsonObj, "y", 0);
                ModelRotation modelrotation = ModelRotation.getModelRotation(i, j);

                if (modelrotation == null) {
                    throw new JsonParseException("Invalid BlockModelRotation x: " + i + ", y: " + j);
                } else {
                    return modelrotation;
                }
            }

            protected String parseModel(JsonObject jsonObj) {
                return JsonUtils.getString(jsonObj, "model");
            }

            protected int parseWeight(JsonObject jsonObj) {
                return JsonUtils.getInt(jsonObj, "weight", 1);
            }
        }
    }

    public static class Variants {
        private final String name;
        private final List<ModelBlockDefinition.Variant> listVariants;

        public Variants(String nameIn, List<ModelBlockDefinition.Variant> listVariantsIn) {
            this.name = nameIn;
            this.listVariants = listVariantsIn;
        }

        public List<ModelBlockDefinition.Variant> getVariants() {
            return this.listVariants;
        }

        public boolean equals(Object p_equals_1_) {
            if (this == p_equals_1_) {
                return true;
            } else if (!(p_equals_1_ instanceof Variants modelblockdefinition$variants)) {
                return false;
            } else {
                return this.name.equals(modelblockdefinition$variants.name) && this.listVariants.equals(modelblockdefinition$variants.listVariants);
            }
        }

        public int hashCode() {
            int i = this.name.hashCode();
            i = 31 * i + this.listVariants.hashCode();
            return i;
        }
    }
}
