package net.minecraft.client.renderer.block.model;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import net.minecraft.client.renderer.texture.TextureAtlasSprite;
import net.minecraft.client.renderer.texture.TextureMap;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.ResourceLocation;
import org.lwjgl.util.vector.Vector3f;

import java.util.List;
import java.util.Map;

public class ItemModelGenerator {
    public static final List<String> LAYERS = Lists.newArrayList("layer0", "layer1", "layer2", "layer3", "layer4");

    public ModelBlock makeItemModel(TextureMap textureMapIn, ModelBlock blockModel) {
        Map<String, String> map = Maps.newHashMap();
        List<BlockPart> list = Lists.newArrayList();

        for (int i = 0; i < LAYERS.size(); ++i) {
            String s = LAYERS.get(i);

            if (!blockModel.isTexturePresent(s)) {
                break;
            }

            String s1 = blockModel.resolveTextureName(s);
            map.put(s, s1);
            TextureAtlasSprite textureatlassprite = textureMapIn.getAtlasSprite((new ResourceLocation(s1)).toString());
            list.addAll(this.getBlockParts(i, s, textureatlassprite));
        }

        if (list.isEmpty()) {
            return null;
        } else {
            map.put("particle", blockModel.isTexturePresent("particle") ? blockModel.resolveTextureName("particle") : map.get("layer0"));
            return new ModelBlock(list, map, false, false, blockModel.getAllTransforms());
        }
    }

    private List<BlockPart> getBlockParts(int tintIndex, String texture, TextureAtlasSprite sprite) {
        Map<EnumFacing, BlockPartFace> map = Maps.newHashMap();
        map.put(EnumFacing.SOUTH, new BlockPartFace(null, tintIndex, texture, new BlockFaceUV(new float[]{0.0F, 0.0F, 16.0F, 16.0F}, 0)));
        map.put(EnumFacing.NORTH, new BlockPartFace(null, tintIndex, texture, new BlockFaceUV(new float[]{16.0F, 0.0F, 0.0F, 16.0F}, 0)));
        List<BlockPart> list = Lists.newArrayList();
        list.add(new BlockPart(new Vector3f(0.0F, 0.0F, 7.5F), new Vector3f(16.0F, 16.0F, 8.5F), map, null, true));
        list.addAll(this.getBlockParts(sprite, texture, tintIndex));
        return list;
    }

    private List<BlockPart> getBlockParts(TextureAtlasSprite sprite, String texture, int tintIndex) {
        float f = (float) sprite.getIconWidth();
        float f1 = (float) sprite.getIconHeight();
        List<BlockPart> list = Lists.newArrayList();

        for (ItemModelGenerator.Span itemmodelgenerator$span : this.getSpans(sprite)) {
            float f2 = 0.0F;
            float f3 = 0.0F;
            float f4 = 0.0F;
            float f5 = 0.0F;
            float f6 = 0.0F;
            float f7 = 0.0F;
            float f8 = 0.0F;
            float f9 = 0.0F;
            float f10 = 0.0F;
            float f11 = 0.0F;
            float f12 = (float) itemmodelgenerator$span.getMin();
            float f13 = (float) itemmodelgenerator$span.getMax();
            float f14 = (float) itemmodelgenerator$span.getAnchor();
            ItemModelGenerator.SpanFacing itemmodelgenerator$spanfacing = itemmodelgenerator$span.getFacing();

            switch (itemmodelgenerator$spanfacing) {
                case UP:
                    f6 = f12;
                    f2 = f12;
                    f4 = f7 = f13 + 1.0F;
                    f8 = f14;
                    f3 = f14;
                    f9 = f14;
                    f5 = f14;
                    f10 = 16.0F / f;
                    f11 = 16.0F / (f1 - 1.0F);
                    break;

                case DOWN:
                    f9 = f14;
                    f8 = f14;
                    f6 = f12;
                    f2 = f12;
                    f4 = f7 = f13 + 1.0F;
                    f3 = f14 + 1.0F;
                    f5 = f14 + 1.0F;
                    f10 = 16.0F / f;
                    f11 = 16.0F / (f1 - 1.0F);
                    break;

                case LEFT:
                    f6 = f14;
                    f2 = f14;
                    f7 = f14;
                    f4 = f14;
                    f9 = f12;
                    f3 = f12;
                    f5 = f8 = f13 + 1.0F;
                    f10 = 16.0F / (f - 1.0F);
                    f11 = 16.0F / f1;
                    break;

                case RIGHT:
                    f7 = f14;
                    f6 = f14;
                    f2 = f14 + 1.0F;
                    f4 = f14 + 1.0F;
                    f9 = f12;
                    f3 = f12;
                    f5 = f8 = f13 + 1.0F;
                    f10 = 16.0F / (f - 1.0F);
                    f11 = 16.0F / f1;
            }

            float f15 = 16.0F / f;
            float f16 = 16.0F / f1;
            f2 = f2 * f15;
            f4 = f4 * f15;
            f3 = f3 * f16;
            f5 = f5 * f16;
            f3 = 16.0F - f3;
            f5 = 16.0F - f5;
            f6 = f6 * f10;
            f7 = f7 * f10;
            f8 = f8 * f11;
            f9 = f9 * f11;
            Map<EnumFacing, BlockPartFace> map = Maps.newHashMap();
            map.put(itemmodelgenerator$spanfacing.getFacing(), new BlockPartFace(null, tintIndex, texture, new BlockFaceUV(new float[]{f6, f8, f7, f9}, 0)));

            switch (itemmodelgenerator$spanfacing) {
                case UP:
                    list.add(new BlockPart(new Vector3f(f2, f3, 7.5F), new Vector3f(f4, f3, 8.5F), map, null, true));
                    break;

                case DOWN:
                    list.add(new BlockPart(new Vector3f(f2, f5, 7.5F), new Vector3f(f4, f5, 8.5F), map, null, true));
                    break;

                case LEFT:
                    list.add(new BlockPart(new Vector3f(f2, f3, 7.5F), new Vector3f(f2, f5, 8.5F), map, null, true));
                    break;

                case RIGHT:
                    list.add(new BlockPart(new Vector3f(f4, f3, 7.5F), new Vector3f(f4, f5, 8.5F), map, null, true));
            }
        }

        return list;
    }

    private List<ItemModelGenerator.Span> getSpans(TextureAtlasSprite sprite) {
        int i = sprite.getIconWidth();
        int j = sprite.getIconHeight();
        List<ItemModelGenerator.Span> list = Lists.newArrayList();

        for (int k = 0; k < sprite.getFrameCount(); ++k) {
            int[] aint = sprite.getFrameTextureData(k)[0];

            for (int l = 0; l < j; ++l) {
                for (int i1 = 0; i1 < i; ++i1) {
                    boolean flag = !this.isTransparent(aint, i1, l, i, j);
                    this.checkTransition(ItemModelGenerator.SpanFacing.UP, list, aint, i1, l, i, j, flag);
                    this.checkTransition(ItemModelGenerator.SpanFacing.DOWN, list, aint, i1, l, i, j, flag);
                    this.checkTransition(ItemModelGenerator.SpanFacing.LEFT, list, aint, i1, l, i, j, flag);
                    this.checkTransition(ItemModelGenerator.SpanFacing.RIGHT, list, aint, i1, l, i, j, flag);
                }
            }
        }

        return list;
    }

    private void checkTransition(ItemModelGenerator.SpanFacing facing, List<ItemModelGenerator.Span> spanList, int[] frameData, int pixelX, int pixelY, int spriteWidth, int spriteHeight, boolean transparent) {
        boolean flag = this.isTransparent(frameData, pixelX + facing.getXOffset(), pixelY + facing.getYOffset(), spriteWidth, spriteHeight) && transparent;

        if (flag) {
            this.createOrExpandSpan(spanList, facing, pixelX, pixelY);
        }
    }

    private void createOrExpandSpan(List<ItemModelGenerator.Span> spanList, ItemModelGenerator.SpanFacing facing, int pixelX, int pixelY) {
        ItemModelGenerator.Span itemmodelgenerator$span = null;

        for (ItemModelGenerator.Span itemmodelgenerator$span1 : spanList) {
            if (itemmodelgenerator$span1.getFacing() == facing) {
                int i = facing.isHorizontal() ? pixelY : pixelX;

                if (itemmodelgenerator$span1.getAnchor() == i) {
                    itemmodelgenerator$span = itemmodelgenerator$span1;
                    break;
                }
            }
        }

        int j = facing.isHorizontal() ? pixelY : pixelX;
        int k = facing.isHorizontal() ? pixelX : pixelY;

        if (itemmodelgenerator$span == null) {
            spanList.add(new ItemModelGenerator.Span(facing, k, j));
        } else {
            itemmodelgenerator$span.expand(k);
        }
    }

    private boolean isTransparent(int[] frameData, int x, int y, int width, int height) {
        return x < 0 || y < 0 || x >= width || y >= height || (frameData[y * width + x] >> 24 & 255) == 0;
    }

    enum SpanFacing {
        UP(EnumFacing.UP, 0, -1),
        DOWN(EnumFacing.DOWN, 0, 1),
        LEFT(EnumFacing.EAST, -1, 0),
        RIGHT(EnumFacing.WEST, 1, 0);

        private final EnumFacing facing;
        private final int xOffset;
        private final int yOffset;

        SpanFacing(EnumFacing facing, int xOffsetIn, int yOffsetIn) {
            this.facing = facing;
            this.xOffset = xOffsetIn;
            this.yOffset = yOffsetIn;
        }

        public EnumFacing getFacing() {
            return this.facing;
        }

        public int getXOffset() {
            return this.xOffset;
        }

        public int getYOffset() {
            return this.yOffset;
        }

        private boolean isHorizontal() {
            return this == DOWN || this == UP;
        }
    }

    static class Span {
        private final ItemModelGenerator.SpanFacing spanFacing;
        private final int anchor;
        private int min;
        private int max;

        public Span(ItemModelGenerator.SpanFacing spanFacingIn, int minIn, int maxIn) {
            this.spanFacing = spanFacingIn;
            this.min = minIn;
            this.max = minIn;
            this.anchor = maxIn;
        }

        public void expand(int factor) {
            if (factor < this.min) {
                this.min = factor;
            } else if (factor > this.max) {
                this.max = factor;
            }
        }

        public ItemModelGenerator.SpanFacing getFacing() {
            return this.spanFacing;
        }

        public int getMin() {
            return this.min;
        }

        public int getMax() {
            return this.max;
        }

        public int getAnchor() {
            return this.anchor;
        }
    }
}
