package net.minecraft.client.network;

import com.google.common.base.MoreObjects;
import com.mojang.authlib.GameProfile;
import net.minecraft.client.Minecraft;
import net.minecraft.client.resources.DefaultPlayerSkin;
import net.minecraft.network.play.server.S38PacketPlayerListItem;
import net.minecraft.scoreboard.ScorePlayerTeam;
import net.minecraft.util.IChatComponent;
import net.minecraft.util.ResourceLocation;
import net.minecraft.world.WorldSettings;

public class NetworkPlayerInfo {
    private final GameProfile gameProfile;
    private WorldSettings.GameType gameType;
    private int responseTime;
    private boolean playerTexturesLoaded = false;
    private ResourceLocation locationSkin;
    private ResourceLocation locationCape;
    private String skinType;
    private IChatComponent displayName;
    private int lastHealth = 0;
    private int displayHealth = 0;
    private long lastHealthTime = 0L;
    private long healthBlinkTime = 0L;
    private long renderVisibilityId = 0L;

    public NetworkPlayerInfo(GameProfile profile) {
        this.gameProfile = profile;
    }

    public NetworkPlayerInfo(S38PacketPlayerListItem.AddPlayerData playerData) {
        this.gameProfile = playerData.getProfile();
        this.gameType = playerData.getGameMode();
        this.responseTime = playerData.getPing();
        this.displayName = playerData.getDisplayName();
    }

    public GameProfile getGameProfile() {
        return this.gameProfile;
    }

    public WorldSettings.GameType getGameType() {
        return this.gameType;
    }

    protected void setGameType(WorldSettings.GameType gameMode) {
        this.gameType = gameMode;
    }

    public int getResponseTime() {
        return this.responseTime;
    }

    protected void setResponseTime(int latency) {
        this.responseTime = latency;
    }

    public boolean hasLocationSkin() {
        return this.locationSkin != null;
    }

    public String getSkinType() {
        return this.skinType == null ? DefaultPlayerSkin.getSkinType(this.gameProfile.getId()) : this.skinType;
    }

    public ResourceLocation getLocationSkin() {
        if (this.locationSkin == null) {
            this.loadPlayerTextures();
        }

        return MoreObjects.firstNonNull(this.locationSkin, DefaultPlayerSkin.getDefaultSkin(this.gameProfile.getId()));
    }

    public ResourceLocation getLocationCape() {
        if (this.locationCape == null) {
            this.loadPlayerTextures();
        }

        return this.locationCape;
    }

    public ScorePlayerTeam getPlayerTeam() {
        return Minecraft.getMinecraft().theWorld.getScoreboard().getPlayersTeam(this.getGameProfile().getName());
    }

    protected void loadPlayerTextures() {
        synchronized (this) {
            if (!this.playerTexturesLoaded) {
                this.playerTexturesLoaded = true;
                Minecraft.getMinecraft().getSkinManager().loadProfileTextures(this.gameProfile, (typeIn, location, profileTexture) -> {
                    switch (typeIn) {
                        case SKIN -> {
                            NetworkPlayerInfo.this.locationSkin = location;
                            NetworkPlayerInfo.this.skinType = profileTexture.getMetadata("model");
                            if (NetworkPlayerInfo.this.skinType == null) {
                                NetworkPlayerInfo.this.skinType = "default";
                            }
                        }
                        case CAPE -> NetworkPlayerInfo.this.locationCape = location;
                    }
                }, true);
            }
        }
    }

    public IChatComponent getDisplayName() {
        return this.displayName;
    }

    public void setDisplayName(IChatComponent displayNameIn) {
        this.displayName = displayNameIn;
    }

    public int getLastHealth() {
        return this.lastHealth;
    }

    public void setLastHealth(int value) {
        this.lastHealth = value;
    }

    public int getDisplayHealth() {
        return this.displayHealth;
    }

    public void setDisplayHealth(int value) {
        this.displayHealth = value;
    }

    public long getLastHealthTime() {
        return this.lastHealthTime;
    }

    public void setLastHealthTime(long value) {
        this.lastHealthTime = value;
    }

    public long getHealthBlinkTime() {
        return this.healthBlinkTime;
    }

    public void setHealthBlinkTime(long value) {
        this.healthBlinkTime = value;
    }

    public long getRenderVisibilityId() {
        return this.renderVisibilityId;
    }

    public void setRenderVisibilityId(long value) {
        this.renderVisibilityId = value;
    }

    // Start of BloomClient

    public String getName() {
        if (this.displayName != null)
            return displayName.getFormattedText();

        return ScorePlayerTeam.formatPlayerName(this.getPlayerTeam(), this.gameProfile.getName());
    }

    // End of BloomClient

}
