package net.minecraft.client.gui;

import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.spectator.ISpectatorMenuObject;
import net.minecraft.client.gui.spectator.ISpectatorMenuRecipient;
import net.minecraft.client.gui.spectator.SpectatorMenu;
import net.minecraft.client.gui.spectator.categories.SpectatorDetails;
import net.minecraft.client.option.options.devices.KeyBinding;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.RenderHelper;
import net.minecraft.util.MathHelper;
import net.minecraft.util.ResourceLocation;

public class GuiSpectator extends Gui implements ISpectatorMenuRecipient {
    public static final ResourceLocation SPECTATOR_WIDGETS = new ResourceLocation("textures/gui/spectator_widgets.png");
    private static final ResourceLocation WIDGETS = new ResourceLocation("textures/gui/widgets.png");
    private final Minecraft mc;
    private long lastSelectionTime;
    private SpectatorMenu menu;

    public GuiSpectator(Minecraft mcIn) {
        this.mc = mcIn;
    }

    public void onHotbarSelected(int index) {
        this.lastSelectionTime = Minecraft.getSystemTime();

        if (this.menu != null) {
            this.menu.selectSlot(index);
        } else {
            this.menu = new SpectatorMenu(this);
        }
    }

    private float getHotbarAlpha() {
        long i = this.lastSelectionTime - Minecraft.getSystemTime() + 5000L;
        return MathHelper.clamp_float((float) i / 2000.0F, 0.0F, 1.0F);
    }

    public void renderTooltip(ScaledResolution sc, float partialTicks) {
        if (this.menu != null) {
            float f = this.getHotbarAlpha();

            if (f <= 0.0F) {
                this.menu.exit();
            } else {
                int i = sc.getScaledWidth() / 2;
                float f1 = this.zLevel;
                this.zLevel = -90.0F;
                float f2 = (float) sc.getScaledHeight() - 22.0F * f;
                SpectatorDetails spectatordetails = this.menu.getCurrentPage();
                this.renderPage(sc, f, i, f2, spectatordetails);
                this.zLevel = f1;
            }
        }

        // render 2d event
    }

    protected void renderPage(ScaledResolution sc, float alpha, int x, float y, SpectatorDetails details) {
        GlStateManager.enableRescaleNormal();
        GlStateManager.enableBlend();
        GlStateManager.tryBlendFuncSeparate(770, 771, 1, 0);
        GlStateManager.color(1.0F, 1.0F, 1.0F, alpha);
        this.mc.getTextureManager().bindTexture(WIDGETS);
        this.drawTexturedModalRect((float) (x - 91), y, 0, 0, 182, 22);

        if (details.getSelectedSlot() >= 0) {
            this.drawTexturedModalRect((float) (x - 91 - 1 + details.getSelectedSlot() * 20), y - 1.0F, 0, 22, 24, 22);
        }

        RenderHelper.enableGUIStandardItemLighting();

        for (int i = 0; i < 9; ++i) {
            this.renderSlot(i, sc.getScaledWidth() / 2 - 90 + i * 20 + 2, y + 3.0F, alpha, details.getObject(i));
        }

        RenderHelper.disableStandardItemLighting();
        GlStateManager.disableRescaleNormal();
        GlStateManager.disableBlend();
    }

    private void renderSlot(int index, int x, float y, float alpha, ISpectatorMenuObject menu) {
        this.mc.getTextureManager().bindTexture(SPECTATOR_WIDGETS);

        if (menu != SpectatorMenu.EMPTY_SLOT) {
            int i = (int) (alpha * 255.0F);
            GlStateManager.pushMatrix();
            GlStateManager.translate((float) x, y, 0.0F);
            float f = menu.isEnabled() ? 1.0F : 0.25F;
            GlStateManager.color(f, f, f, alpha);
            menu.renderIcon(f, i);
            GlStateManager.popMatrix();
            String s = String.valueOf(Minecraft.getKeyDisplayString(KeyBinding.keyBindsHotbar[index].getKeyCode()));

            if (i > 3 && menu.isEnabled()) {
                this.mc.minecraftFontRendererObj.drawStringWithShadow(s, (float) (x + 19 - 2 - this.mc.minecraftFontRendererObj.getStringWidth(s)), y + 6.0F + 3.0F, 16777215 + (i << 24));
            }
        }
    }

    public void renderSelectedItem(ScaledResolution sc) {
        int i = (int) (this.getHotbarAlpha() * 255.0F);

        if (i > 3 && this.menu != null) {
            ISpectatorMenuObject ispectatormenuobject = this.menu.getSelectedItem();
            String s = ispectatormenuobject != SpectatorMenu.EMPTY_SLOT ? ispectatormenuobject.getSpectatorName().getFormattedText() : this.menu.getSelectedCategory().getPrompt().getFormattedText();

            if (s != null) {
                int j = (sc.getScaledWidth() - this.mc.minecraftFontRendererObj.getStringWidth(s)) / 2;
                int k = sc.getScaledHeight() - 35;
                GlStateManager.pushMatrix();
                GlStateManager.enableBlend();
                GlStateManager.tryBlendFuncSeparate(770, 771, 1, 0);
                this.mc.minecraftFontRendererObj.drawStringWithShadow(s, (float) j, (float) k, 16777215 + (i << 24));
                GlStateManager.disableBlend();
                GlStateManager.popMatrix();
            }
        }
    }

    public void onSpectatorMenuClosed(SpectatorMenu menu) {
        this.menu = null;
        this.lastSelectionTime = 0L;
    }

    public boolean isMenuActive() {
        return this.menu != null;
    }

    public void onMouseScroll(int increment) {
        int i;

        for (i = this.menu.getSelectedSlot() + increment; i >= 0 && i <= 8 && (this.menu.getItem(i) == SpectatorMenu.EMPTY_SLOT || !this.menu.getItem(i).isEnabled()); i += increment) {
        }

        if (i >= 0 && i <= 8) {
            this.menu.selectSlot(i);
            this.lastSelectionTime = Minecraft.getSystemTime();
        }
    }

    public void onMiddleClick() {
        this.lastSelectionTime = Minecraft.getSystemTime();

        if (this.isMenuActive()) {
            int i = this.menu.getSelectedSlot();

            if (i != -1) {
                this.menu.selectSlot(i);
            }
        } else {
            this.menu = new SpectatorMenu(this);
        }
    }
}
