package net.minecraft.client.gui;

import com.google.common.base.MoreObjects;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import net.minecraft.client.Minecraft;

import java.util.List;

public class GuiPageButtonList extends GuiListExtended {
    private final List<GuiEntry> entries = Lists.newArrayList();
    private final Int2ObjectOpenHashMap<Gui> componentMap = new Int2ObjectOpenHashMap<>();
    private final List<GuiTextField> editBoxes = Lists.newArrayList();
    private final GuiListEntry[][] pages;
    private final GuiResponder responder;
    private int page;
    private Gui focusedControl;

    public GuiPageButtonList(Minecraft mcIn, int widthIn, int heightIn, int topIn, int bottomIn, int slotHeightIn, GuiResponder responder, GuiListEntry[]... pages) {
        super(mcIn, widthIn, heightIn, topIn, bottomIn, slotHeightIn);
        this.responder = responder;
        this.pages = pages;
        this.centerListVertically = false;
        this.populateComponents();
        this.populateEntries();
    }

    private void populateComponents() {
        for (GuiListEntry[] aguipagebuttonlist$guilistentry : this.pages) {
            for (int i = 0; i < aguipagebuttonlist$guilistentry.length; i += 2) {
                GuiListEntry guipagebuttonlist$guilistentry = aguipagebuttonlist$guilistentry[i];
                GuiListEntry guipagebuttonlist$guilistentry1 = i < aguipagebuttonlist$guilistentry.length - 1 ? aguipagebuttonlist$guilistentry[i + 1] : null;
                Gui gui = this.createEntry(guipagebuttonlist$guilistentry, 0, guipagebuttonlist$guilistentry1 == null);
                Gui gui1 = this.createEntry(guipagebuttonlist$guilistentry1, 160, guipagebuttonlist$guilistentry == null);
                GuiEntry guipagebuttonlist$guientry = new GuiEntry(gui, gui1);
                this.entries.add(guipagebuttonlist$guientry);

                if (guipagebuttonlist$guilistentry != null && gui != null) {
                    this.componentMap.put(guipagebuttonlist$guilistentry.getId(), gui);

                    if (gui instanceof GuiTextField) {
                        this.editBoxes.add((GuiTextField) gui);
                    }
                }

                if (guipagebuttonlist$guilistentry1 != null && gui1 != null) {
                    this.componentMap.put(guipagebuttonlist$guilistentry1.getId(), gui1);

                    if (gui1 instanceof GuiTextField) {
                        this.editBoxes.add((GuiTextField) gui1);
                    }
                }
            }
        }
    }

    private void populateEntries() {
        this.entries.clear();

        for (int i = 0; i < this.pages[this.page].length; i += 2) {
            GuiListEntry guipagebuttonlist$guilistentry = this.pages[this.page][i];
            GuiListEntry guipagebuttonlist$guilistentry1 = i < this.pages[this.page].length - 1 ? this.pages[this.page][i + 1] : null;
            Gui gui = this.componentMap.get(guipagebuttonlist$guilistentry.getId());
            Gui gui1 = guipagebuttonlist$guilistentry1 != null ? this.componentMap.get(guipagebuttonlist$guilistentry1.getId()) : null;
            GuiEntry guipagebuttonlist$guientry = new GuiEntry(gui, gui1);
            this.entries.add(guipagebuttonlist$guientry);
        }
    }

    public int getPage() {
        return this.page;
    }

    public void setPage(int page) {
        if (page != this.page) {
            int i = this.page;
            this.page = page;
            this.populateEntries();
            this.markVisibility(i, page);
            this.amountScrolled = 0.0F;
        }
    }

    public int getPageCount() {
        return this.pages.length;
    }

    public Gui getFocusedControl() {
        return this.focusedControl;
    }

    public void previousPage() {
        if (this.page > 0) {
            this.setPage(this.page - 1);
        }
    }

    public void nextPage() {
        if (this.page < this.pages.length - 1) {
            this.setPage(this.page + 1);
        }
    }

    public Gui getComponent(int id) {
        return this.componentMap.get(id);
    }

    private void markVisibility(int oldPage, int newPage) {
        for (GuiListEntry guipagebuttonlist$guilistentry : this.pages[oldPage]) {
            if (guipagebuttonlist$guilistentry != null) {
                this.setComponentVisibility(this.componentMap.get(guipagebuttonlist$guilistentry.getId()), false);
            }
        }

        for (GuiListEntry guipagebuttonlist$guilistentry1 : this.pages[newPage]) {
            if (guipagebuttonlist$guilistentry1 != null) {
                this.setComponentVisibility(this.componentMap.get(guipagebuttonlist$guilistentry1.getId()), true);
            }
        }
    }

    private void setComponentVisibility(Gui gui, boolean visible) {
        if (gui instanceof GuiButton) {
            ((GuiButton) gui).visible = visible;
        } else if (gui instanceof GuiTextField) {
            ((GuiTextField) gui).setVisible(visible);
        } else if (gui instanceof GuiLabel) {
            ((GuiLabel) gui).visible = visible;
        }
    }

    private Gui createEntry(GuiListEntry entry, int xOffset, boolean fullWidth) {
        return entry instanceof GuiSlideEntry ? this.createSilder(this.width / 2 - 155 + xOffset, 0, (GuiSlideEntry) entry) : (entry instanceof GuiButtonEntry ? this.createButton(this.width / 2 - 155 + xOffset, 0, (GuiButtonEntry) entry) : (entry instanceof EditBoxEntry ? this.createTextField(this.width / 2 - 155 + xOffset, 0, (EditBoxEntry) entry) : (entry instanceof GuiLabelEntry ? this.createLabel(this.width / 2 - 155 + xOffset, 0, (GuiLabelEntry) entry, fullWidth) : null)));
    }

    public void setActive(boolean active) {
        for (GuiEntry guipagebuttonlist$guientry : this.entries) {
            if (guipagebuttonlist$guientry.component1 instanceof GuiButton) {
                ((GuiButton) guipagebuttonlist$guientry.component1).enabled = active;
            }

            if (guipagebuttonlist$guientry.component2 instanceof GuiButton) {
                ((GuiButton) guipagebuttonlist$guientry.component2).enabled = active;
            }
        }
    }

    public boolean mouseClicked(int mouseX, int mouseY, int mouseEvent) {
        boolean flag = super.mouseClicked(mouseX, mouseY, mouseEvent);
        int i = this.getSlotIndexFromScreenCoords(mouseX, mouseY);

        if (i >= 0) {
            GuiEntry guipagebuttonlist$guientry = this.getListEntry(i);

            if (this.focusedControl != guipagebuttonlist$guientry.focusedControl && this.focusedControl != null && this.focusedControl instanceof GuiTextField) {
                ((GuiTextField) this.focusedControl).setFocused(false);
            }

            this.focusedControl = guipagebuttonlist$guientry.focusedControl;
        }

        return flag;
    }

    private GuiSlider createSilder(int x, int y, GuiSlideEntry entry) {
        GuiSlider guislider = new GuiSlider(this.responder, entry.getId(), x, y, entry.getCaption(), entry.getMinValue(), entry.getMaxValue(), entry.getInitalValue(), entry.getFormatter());
        guislider.visible = entry.shouldStartVisible();
        return guislider;
    }

    private GuiListButton createButton(int x, int y, GuiButtonEntry entry) {
        GuiListButton guilistbutton = new GuiListButton(this.responder, entry.getId(), x, y, entry.getCaption(), entry.getInitialValue());
        guilistbutton.visible = entry.shouldStartVisible();
        return guilistbutton;
    }

    private GuiTextField createTextField(int x, int y, EditBoxEntry entry) {
        GuiTextField guitextfield = new GuiTextField(entry.getId(), this.mc.minecraftFontRendererObj, x, y, 150, 20);
        guitextfield.setText(entry.getCaption());
        guitextfield.setGuiResponder(this.responder);
        guitextfield.setVisible(entry.shouldStartVisible());
        guitextfield.setValidator(entry.getFilter());
        return guitextfield;
    }

    private GuiLabel createLabel(int x, int y, GuiLabelEntry entry, boolean fullWidth) {
        GuiLabel guilabel;

        if (fullWidth) {
            guilabel = new GuiLabel(this.mc.minecraftFontRendererObj, entry.getId(), x, y, this.width - x * 2, 20, -1);
        } else {
            guilabel = new GuiLabel(this.mc.minecraftFontRendererObj, entry.getId(), x, y, 150, 20, -1);
        }

        guilabel.visible = entry.shouldStartVisible();
        guilabel.addLine(entry.getCaption());
        guilabel.setCentered();
        return guilabel;
    }

    public void onKeyPressed(char typedChar, int keyCode) {
        if (this.focusedControl instanceof GuiTextField guitextfield) {

            if (!GuiScreen.isKeyComboCtrlV(keyCode)) {
                if (keyCode == 15) {
                    guitextfield.setFocused(false);
                    int k = this.editBoxes.indexOf(this.focusedControl);

                    if (GuiScreen.isShiftKeyDown()) {
                        if (k == 0) {
                            k = this.editBoxes.size() - 1;
                        } else {
                            --k;
                        }
                    } else if (k == this.editBoxes.size() - 1) {
                        k = 0;
                    } else {
                        ++k;
                    }

                    this.focusedControl = this.editBoxes.get(k);
                    guitextfield = (GuiTextField) this.focusedControl;
                    guitextfield.setFocused(true);
                    int l = guitextfield.yPosition + this.slotHeight;
                    int i1 = guitextfield.yPosition;

                    if (l > this.bottom) {
                        this.amountScrolled += (float) (l - this.bottom);
                    } else if (i1 < this.top) {
                        this.amountScrolled = (float) i1;
                    }
                } else {
                    guitextfield.textboxKeyTyped(typedChar, keyCode);
                }
            } else {
                String s = GuiScreen.getClipboardString();
                String[] astring = s.split(";");
                int i = this.editBoxes.indexOf(this.focusedControl);
                int j = i;

                for (String s1 : astring) {
                    this.editBoxes.get(j).setText(s1);

                    if (j == this.editBoxes.size() - 1) {
                        j = 0;
                    } else {
                        ++j;
                    }

                    if (j == i) {
                        break;
                    }
                }
            }
        }
    }

    public GuiEntry getListEntry(int index) {
        return this.entries.get(index);
    }

    public int getSize() {
        return this.entries.size();
    }

    public int getListWidth() {
        return 400;
    }

    protected int getScrollBarX() {
        return super.getScrollBarX() + 32;
    }

    public interface GuiResponder {
        void setEntryValue(int id, boolean value);

        void onTick(int id, float value);

        void setEntryValue(int id, String value);
    }

    public static class EditBoxEntry extends GuiListEntry {
        private final Predicate<String> filter;

        public EditBoxEntry(int id, String caption, boolean startVisible, Predicate<String> filter) {
            super(id, caption, startVisible);
            this.filter = MoreObjects.firstNonNull(filter, s -> true);
        }

        public Predicate<String> getFilter() {
            return this.filter;
        }
    }

    public static class GuiButtonEntry extends GuiListEntry {
        private final boolean initialValue;

        public GuiButtonEntry(int id, String caption, boolean startVisible, boolean initialValue) {
            super(id, caption, startVisible);
            this.initialValue = initialValue;
        }

        public boolean getInitialValue() {
            return this.initialValue;
        }
    }

    public static class GuiEntry implements IGuiListEntry {
        private final Minecraft client = Minecraft.getMinecraft();
        private final Gui component1;
        private final Gui component2;
        private Gui focusedControl;

        public GuiEntry(Gui component1, Gui component2) {
            this.component1 = component1;
            this.component2 = component2;
        }

        public Gui getComponent1() {
            return this.component1;
        }

        public Gui getComponent2() {
            return this.component2;
        }

        public void drawEntry(int slotIndex, int x, int y, int listWidth, int slotHeight, int mouseX, int mouseY, boolean isSelected) {
            this.renderComponent(this.component1, y, mouseX, mouseY, false);
            this.renderComponent(this.component2, y, mouseX, mouseY, false);
        }

        private void renderComponent(Gui component, int y, int mouseX, int mouseY, boolean isSelected) {
            if (component != null) {
                if (component instanceof GuiButton) {
                    this.renderButton((GuiButton) component, y, mouseX, mouseY, isSelected);
                } else if (component instanceof GuiTextField) {
                    this.renderTextField((GuiTextField) component, y, isSelected);
                } else if (component instanceof GuiLabel) {
                    this.renderLabel((GuiLabel) component, y, mouseX, mouseY, isSelected);
                }
            }
        }

        private void renderButton(GuiButton button, int y, int mouseX, int mouseY, boolean isSelected) {
            button.yPosition = y;

            if (!isSelected) {
                button.drawButton(this.client, mouseX, mouseY);
            }
        }

        private void renderTextField(GuiTextField field, int y, boolean isSelected) {
            field.yPosition = y;

            if (!isSelected) {
                field.drawTextBox();
            }
        }

        private void renderLabel(GuiLabel label, int y, int mouseX, int mouseY, boolean isSelected) {
            label.y = y;

            if (!isSelected) {
                label.drawLabel(this.client, mouseX, mouseY);
            }
        }

        public void setSelected(int entryID, int insideLeft, int yPos) {
            this.renderComponent(this.component1, yPos, 0, 0, true);
            this.renderComponent(this.component2, yPos, 0, 0, true);
        }

        public boolean mousePressed(int slotIndex, int mouseX, int mouseY, int mouseEvent, int relativeX, int relativeY) {
            boolean flag = this.clickComponent(this.component1, mouseX, mouseY, mouseEvent);
            boolean flag1 = this.clickComponent(this.component2, mouseX, mouseY, mouseEvent);
            return flag || flag1;
        }

        private boolean clickComponent(Gui component, int mouseX, int mouseY, int mouseEvent) {
            if (component == null) {
                return false;
            } else if (component instanceof GuiButton) {
                return this.clickButton((GuiButton) component, mouseX, mouseY, mouseEvent);
            } else {
                if (component instanceof GuiTextField) {
                    this.clickTextField((GuiTextField) component, mouseX, mouseY, mouseEvent);
                }

                return false;
            }
        }

        private boolean clickButton(GuiButton button, int mouseX, int mouseY, int event) {
            boolean flag = button.mousePressed(this.client, mouseX, mouseY);

            if (flag) {
                this.focusedControl = button;
            }

            return flag;
        }

        private void clickTextField(GuiTextField field, int mouseX, int mouseY, int event) {
            field.mouseClicked(mouseX, mouseY, event);

            if (field.isFocused()) {
                this.focusedControl = field;
            }
        }

        public void mouseReleased(int slotIndex, int x, int y, int mouseEvent, int relativeX, int relativeY) {
            this.releaseComponent(this.component1, x, y, mouseEvent);
            this.releaseComponent(this.component2, x, y, mouseEvent);
        }

        private void releaseComponent(Gui component, int x, int y, int mouseEvent) {
            if (component != null) {
                if (component instanceof GuiButton) {
                    this.releaseButton((GuiButton) component, x, y, mouseEvent);
                }
            }
        }

        private void releaseButton(GuiButton button, int x, int y, int event) {
            button.mouseReleased(x, y);
        }
    }

    public static class GuiLabelEntry extends GuiListEntry {
        public GuiLabelEntry(int id, String caption, boolean startVisible) {
            super(id, caption, startVisible);
        }
    }

    public static class GuiListEntry {
        private final int id;
        private final String caption;
        private final boolean startVisible;

        public GuiListEntry(int id, String caption, boolean startVisible) {
            this.id = id;
            this.caption = caption;
            this.startVisible = startVisible;
        }

        public int getId() {
            return this.id;
        }

        public String getCaption() {
            return this.caption;
        }

        public boolean shouldStartVisible() {
            return this.startVisible;
        }
    }

    public static class GuiSlideEntry extends GuiListEntry {
        private final GuiSlider.FormatHelper formatter;
        private final float minValue;
        private final float maxValue;
        private final float initialValue;

        public GuiSlideEntry(int id, String caption, boolean startVisible, GuiSlider.FormatHelper formatter, float minValue, float maxValue, float initialValue) {
            super(id, caption, startVisible);
            this.formatter = formatter;
            this.minValue = minValue;
            this.maxValue = maxValue;
            this.initialValue = initialValue;
        }

        public GuiSlider.FormatHelper getFormatter() {
            return this.formatter;
        }

        public float getMinValue() {
            return this.minValue;
        }

        public float getMaxValue() {
            return this.maxValue;
        }

        public float getInitalValue() {
            return this.initialValue;
        }
    }
}
