package net.lenni0451.lambdaevents.generator;

import net.lenni0451.lambdaevents.AHandler;
import net.lenni0451.lambdaevents.EventHandler;
import net.lenni0451.lambdaevents.IGenerator;
import net.lenni0451.lambdaevents.handler.methodhandle.MethodHandleHandler;
import net.lenni0451.lambdaevents.handler.methodhandle.VirtualMethodHandleHandler;
import net.lenni0451.lambdaevents.utils.LookupUtils;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * The {@link IGenerator} implementation which calls the handler method using method handles.
 */
public class MethodHandleGenerator implements IGenerator {

    private final MethodHandles.Lookup lookup;

    /**
     * Use the {@link MethodHandles.Lookup} of the current {@link ClassLoader}.
     */
    public MethodHandleGenerator() {
        this(MethodHandles.lookup());
    }

    /**
     * @param lookup The {@link MethodHandles.Lookup} to use
     */
    public MethodHandleGenerator(final MethodHandles.Lookup lookup) {
        this.lookup = lookup;
    }

    @Override
    public AHandler generate(Class<?> owner, Object instance, EventHandler annotation, Method method, Class<?> arg) throws IllegalAccessException, ClassNotFoundException, InvocationTargetException, NoSuchMethodException {
        MethodHandle handle = this.getHandle(owner, instance, method);
        return new MethodHandleHandler(owner, instance, annotation, handle);
    }

    @Override
    public AHandler generateVirtual(Class<?> owner, Object instance, EventHandler annotation, Method method) throws IllegalAccessException, ClassNotFoundException, InvocationTargetException, NoSuchMethodException {
        MethodHandle handle = this.getHandle(owner, instance, method);
        return new VirtualMethodHandleHandler(owner, instance, annotation, handle);
    }

    private MethodHandle getHandle(final Class<?> owner, final Object instance, final Method method) throws IllegalAccessException, ClassNotFoundException, InvocationTargetException, NoSuchMethodException {
        MethodHandle handle = LookupUtils.resolveLookup(this.lookup, owner).unreflect(method); //Resolve the lookup that it can access the method and unreflect it
        if (instance != null)
            handle = handle.bindTo(instance); //If the handler is not static bind the instance to the method handle
        return handle;
    }

}
