package fag.ml.rise.util;

import fag.ml.util.Constants;
import fag.ml.rise.Main;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.tree.*;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;

/**
 * @author markuss
 */
public class RisePatchUtil implements Constants {
    public static void patchBooleanMethodAndSubclasses(String packagePrefix, String requiredString) {
        String baseSuperName;
        var booleanMethodDesc = "()Z";

        ClassNode targetClass = null;
        MethodNode targetMethod = null;

        for (var node : Main.NODES.values()) {
            if (!node.name.startsWith(packagePrefix)) continue;

            var contains = false;
            for (var m : node.methods) {
                for (var insn : m.instructions) {
                    if (insn instanceof LdcInsnNode ldc && ldc.cst instanceof String str && str.contains(requiredString)) {
                        contains = true;
                        break;
                    }
                }
                if (contains) break;
            }

            if (!contains) continue;

            var boolMethods = node.methods.stream()
                    .filter(m -> m.desc.equals(booleanMethodDesc))
                    .toList();

            if (boolMethods.size() == 1) {
                targetClass = node;
                targetMethod = boolMethods.getFirst();
                break;
            }
        }

        if (targetClass == null || targetMethod == null) {
            LOGGER.info("no matching class with single boolean method found");
            return;
        }

        baseSuperName = targetClass.superName;

        for (var node : Main.NODES.values()) {
            if (node.superName != null && node.superName.equals(baseSuperName)) {
                var boolMethods = node.methods.stream()
                        .filter(m -> m.desc.equals(booleanMethodDesc))
                        .toList();

                if (boolMethods.size() == 1) {
                    LOGGER.info("found class: " + node.name);
                    var m = boolMethods.getFirst();
                    LOGGER.info("found method: " + m.name + m.desc);
                    var subPatch = new InsnList();
                    subPatch.add(new InsnNode(Opcodes.ICONST_0));
                    subPatch.add(new InsnNode(Opcodes.IRETURN));
                    m.instructions = subPatch;
                    LOGGER.info("patched method: " + m.name + m.desc);
                }
            }
        }
    }

    private static List<ClassNode> getAllClassesInPackage(String packagePath) {
        List<ClassNode> matchingClasses = new ArrayList<>();

        for (var entry : Main.NODES.entrySet()) {
            var classNode = entry.getValue();

            if (classNode.name.startsWith(packagePath)) {
                matchingClasses.add(classNode);
            }
        }

        return matchingClasses;
    }

    public static void patchUnsafeMemoryWipe() {
        for (var node : Main.NODES.values()) {
            var methods = node.methods;
            for (int i = 0; i < methods.size(); i++) {
                var method = methods.get(i);

                var containsTheUnsafe = false;
                for (var insn : method.instructions) {
                    //gets the crash method
                    if (insn instanceof LdcInsnNode ldc && ldc.cst instanceof String str && str.contains("theUnsafe")) {
                        LOGGER.info("found class " + node.name);
                        containsTheUnsafe = true;
                        break;
                    }
                }

                if (containsTheUnsafe) {
                    LOGGER.info("found method " + method.name + method.desc);
                    method.instructions = new InsnList();
                    method.instructions.add(new InsnNode(Opcodes.RETURN));
                    LOGGER.info("patched method " + method.name + method.desc);

                    if (i > 0) {
                        var prev = methods.get(i - 1);
                        //gets the hang jvm method
                        LOGGER.info("found method " + prev.name + prev.desc);
                        prev.instructions = new InsnList();
                        prev.instructions.add(new InsnNode(Opcodes.RETURN));
                        LOGGER.info("patched method " + prev.name + prev.desc);
                    }

                    break;
                }
            }
        }
    }

    public static void patchS2C() {
        var allClasses = getAllClassesInPackage("rip/vantage/commons/packet/impl/server/protection");

        for (var classNode : allClasses) {
            if (hasCorrectFieldTypes(classNode)) {
                var methodNodes = classNode.methods;

                for (int i = 0; i < methodNodes.size(); i++) {
                    var method = methodNodes.get(i);
                    var iterator = method.instructions.iterator();

                    var isTargetMethod = false;

                    while (iterator.hasNext()) {
                        var insn = iterator.next();

                        if (insn instanceof TypeInsnNode typeInsn && typeInsn.getOpcode() == Opcodes.NEW && typeInsn.desc.equals("org/json/JSONObject")) {
                            isTargetMethod = true;
                            break;
                        }
                    }

                    if (isTargetMethod) {
                        LOGGER.info("Found S2C packet");
                        if (i + 1 < methodNodes.size()) {
                            var nextMethod = methodNodes.get(i + 1);
                            LOGGER.info("Found method " + nextMethod.name);

                            var subPatch = new InsnList();
                            subPatch.add(new InsnNode(Opcodes.ICONST_1));
                            subPatch.add(new InsnNode(Opcodes.IRETURN));
                            nextMethod.instructions = subPatch;
                            LOGGER.info("Patched method " + nextMethod.name);
                        }
                    }
                }
            }
        }
    }

    public static void patchWebsocketConnections() {
        var allClasses = getAllClassesInPackage("rip/vantage/");

        for (var classNode : allClasses) {
            patchStringsInClass(classNode);
            patchMethodUriCreation(classNode);
        }
    }

    public static void patchAntiDebuggingNew() {
        var allClasses = getAllClassesInPackage("rip/vantage/network/core");

        MethodNode targetMethod = getInit(allClasses);

        if (targetMethod != null) {
            LOGGER.info("Found target method. Proceeding with patching...");
            var it = targetMethod.instructions.iterator();

            while (it.hasNext()) {
                var insn = it.next();

                patchJavaAgentChecks(insn, it);
                patchFishAgentChecks(insn, it);
                patchSysExitCalls(insn, it);
            }
        }
    }

    private static MethodNode getInit(List<ClassNode> allClasses) {
        MethodNode targetMethod = null;

        for (var classNode : allClasses) {
            for (var methodNode : classNode.methods) {
                for (AbstractInsnNode insn : methodNode.instructions) {
                    if (insn.getOpcode() == Opcodes.LDC) {
                        assert insn instanceof LdcInsnNode;
                        LdcInsnNode ldcInsn = (LdcInsnNode) insn;
                        if (ldcInsn.cst.equals("New Session")) {
                            targetMethod = methodNode;
                        }
                    }
                }
            }
        }
        return targetMethod;
    }

    private static void patchSysExitCalls(AbstractInsnNode insn, ListIterator<AbstractInsnNode> it) {
        if (insn.getOpcode() == Opcodes.INVOKESTATIC) {
            var methodInsn = (MethodInsnNode) insn;
            if (methodInsn.owner.equals("java/lang/System") && methodInsn.name.equals("exit") && methodInsn.desc.equals("(I)V")) {
                LOGGER.info("Patching System.exit(1) call. Removing exit call and argument.");
                // Remove the INVOKESTATIC instruction
                it.remove();
                // Also remove the previous instruction that pushes the argument (usually ICONST_1)
                AbstractInsnNode prev = insn.getPrevious();
                if (prev != null && (prev.getOpcode() >= Opcodes.ICONST_M1 && prev.getOpcode() <= Opcodes.ICONST_5 || prev.getOpcode() == Opcodes.BIPUSH || prev.getOpcode() == Opcodes.SIPUSH)) {
                    it.previous();
                    it.remove();
                }
            }
        }
    }


    private static void patchFishAgentChecks(AbstractInsnNode insn, ListIterator<AbstractInsnNode> it) {
        if (insn.getOpcode() == Opcodes.INVOKESTATIC) {
            var methodInsn = (MethodInsnNode) insn;
            if (methodInsn.owner.equals("java/lang/Class") && methodInsn.name.equals("forName") && methodInsn.desc.equals("(Ljava/lang/String;)Ljava/lang/Class;")) {
                // Check previous instruction for the class name string
                AbstractInsnNode prev = insn.getPrevious();
                if (prev instanceof LdcInsnNode) {
                    LdcInsnNode ldc = (LdcInsnNode) prev;
                    if ("cc.fish.agent.CookingAgent".equals(ldc.cst)) {
                        LOGGER.info("Patching class loading of 'cc.fish.agent.CookingAgent'. Removing forName() call and its argument.");
                        // Remove the LDC instruction (class name string)
                        it.previous(); // Move iterator back to LDC
                        it.remove();   // Remove LDC
                        it.next();     // Move back to INVOKESTATIC
                        it.remove();   // Remove INVOKESTATIC
                    }
                }
            }
        }
    }

    private static void patchJavaAgentChecks(AbstractInsnNode insn, ListIterator<AbstractInsnNode> it) {
        // Patch out boolean assignments to true for var2 (javaagent check)
        if (insn.getOpcode() == Opcodes.ICONST_1) {
            AbstractInsnNode next = insn.getNext();
            if (next != null && next.getOpcode() == Opcodes.ISTORE) {
                // This might be the boolean assignment var2 = true;
                // You can add more checks here to confirm variable index if you want
                LOGGER.info("Patching boolean assignment to true for javaagent check. NOPing instructions.");
                it.previous(); // Move iterator back to ICONST_1
                it.remove();   // Remove ICONST_1
                it.next();     // Move to ISTORE
                it.remove();   // Remove ISTORE
            }
        }
    }

    private static void patchStringsInClass(ClassNode classNode) {
        for (var field : classNode.fields) {
            if (field.desc.equals("Ljava/lang/String;") && field.value != null) {
                var value = field.value.toString();
                if (value.contains("auth.riseclient.com")) {
                    LOGGER.info("Found string '" + value + "' in field " + field.name);
                    value = value.replace("auth.riseclient.com", "localhost");
                    field.value = value;
                }
            }
        }

        for (var method : classNode.methods) {
            for (var insn : method.instructions) {
                if (insn instanceof LdcInsnNode ldc) {
                    String value = ldc.cst.toString();
                    if (value.contains("auth.riseclient.com")) {
                        LOGGER.info("Found and replacing string '" + value + "' in method " + method.name);
                        value = value.replace("auth.riseclient.com", "localhost");
                        ldc.cst = value;
                    }
                }
            }
        }
    }

    private static boolean hasCorrectFieldTypes(ClassNode classNode) {
        boolean hasBoolean = false, hasDouble = false, hasFloat = false, hasLong = false, hasString = false;

        for (var field : classNode.fields) {
            switch (field.desc) {
                case "Z": // boolean
                    hasBoolean = true;
                    break;
                case "D": // double
                    hasDouble = true;
                    break;
                case "F": // float
                    hasFloat = true;
                    break;
                case "J": // long
                    hasLong = true;
                    break;
                case "Ljava/lang/String;": // String
                    hasString = true;
                    break;
                default:
                    break;
            }
        }

        return hasBoolean && hasDouble && hasFloat && hasLong && hasString;
    }

    private static void patchMethodUriCreation(ClassNode classNode) {
        for (var method : classNode.methods) {
            if (method.desc.equals("()Ljava/net/URI;")) {
                LOGGER.info("Inspecting method " + method.name);

                for (var insn : method.instructions) {
                    if (insn instanceof MethodInsnNode methodInsn) {
                        if (methodInsn.owner.equals("java/net/URI")
                                && methodInsn.name.equals("<init>")
                                && methodInsn.desc.equals("(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;ILjava/lang/String;Ljava/lang/String;Ljava/lang/String;)V")) {

                            LOGGER.info("Found URI constructor call in method " + method.name);

                            for (var currentInsn : method.instructions) {
                                if (currentInsn instanceof VarInsnNode varInsn) {
                                    if (varInsn.getOpcode() == Opcodes.ALOAD) {
                                        if (varInsn.var == 1) {
                                            LOGGER.info("Replacing var1 (v1) with ldc 'ws'");

                                            method.instructions.insertBefore(currentInsn, new LdcInsnNode("ws"));
                                            method.instructions.remove(currentInsn);
                                        } else if (varInsn.var == 2) {
                                            LOGGER.info("Replacing var2 (v2) with ldc 'localhost'");

                                            method.instructions.insertBefore(currentInsn, new LdcInsnNode("localhost"));
                                            method.instructions.remove(currentInsn);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

}