package fag.ml.util;

import fag.ml.Main;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.stream.Collectors;

public class JVMUtil {
    public static void runJarMain(String jarPath, String[] args) throws Exception {
        URL jarUrl = new File(jarPath).toURI().toURL();
        try (URLClassLoader loader = new URLClassLoader(new URL[]{jarUrl}, null)) { // null = no parent
            Class<?> mainClass = JVMUtil.getMainClass(jarPath, loader);
            Method main = mainClass.getMethod("main", String[].class);
            main.invoke(null, (Object) args);
        }
    }

    private static Class<?> getMainClass(String jarPath, URLClassLoader loader) throws IOException, ClassNotFoundException {
        try (JarFile jarFile = new JarFile(jarPath)) {
            Manifest mf = jarFile.getManifest();
            if (mf == null) throw new IllegalStateException("No manifest found in: " + jarPath);
            String mainClass = mf.getMainAttributes().getValue("Main-Class");
            if (mainClass == null) throw new IllegalStateException("No Main-Class in manifest");
            return loader.loadClass(mainClass.trim());
        }
    }

    public static void runInNewJvm(String className, List<URL> classpath, String nativePath) throws Exception {
        String javaBin = System.getProperty("java.home") + File.separator + "bin" + File.separator + "java";

        System.out.println(javaBin);

        String cp = classpath.stream()
                .map(URL::getPath)
                .collect(Collectors.joining(File.pathSeparator));

        List<String> command = new ArrayList<>();
        command.add(javaBin);
        command.add("-noverify");
        command.add("-Xms4096m");
        command.add("-Xmx4g");
        command.add("-Djava.library.path=" + nativePath);
        command.add("-cp");
        command.add(cp);
        command.add(className);

        ProcessBuilder builder = new ProcessBuilder(command);
        builder.inheritIO(); // Forward stdio to console
        Process process = builder.start();
        int exitCode = process.waitFor();

        if (exitCode != 0) {
            throw new RuntimeException("Process exited with code " + exitCode);
        }
    }
}
