package fag.ml.util;

import fag.ml.Main;
import rip.marie.logger.Level;

import java.io.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

public class FileUtil {
    public static void unzip(File zipFile, File destDir) throws IOException {
        try (ZipInputStream zipIn = new ZipInputStream(new FileInputStream(zipFile))) {
            ZipEntry entry;
            while ((entry = zipIn.getNextEntry()) != null) {
                File filePath = new File(destDir, entry.getName());
                if (entry.isDirectory()) {
                    filePath.mkdirs();
                } else {
                    filePath.getParentFile().mkdirs();
                    try (FileOutputStream out = new FileOutputStream(filePath)) {
                        byte[] buffer = new byte[4096];
                        int len;
                        while ((len = zipIn.read(buffer)) != -1) {
                            out.write(buffer, 0, len);
                        }
                    }
                }
                zipIn.closeEntry();
            }
        }
        Main.LOGGER.log(Level.INFO, "unzipped " + zipFile.getName());
    }

    public static void unzipIfNotExists(String zipPath, String destDir) throws IOException {
        File dest = new File(destDir);
        if (dest.exists() && dest.isDirectory() && dest.list().length > 0) {
            return;
        }
        unzip(new File(zipPath), dest);
    }

    public static String sha1(File file) throws IOException {
        try (InputStream fis = new FileInputStream(file)) {
            java.security.MessageDigest digest = java.security.MessageDigest.getInstance("SHA-1");
            byte[] buffer = new byte[8192];
            int read;
            while ((read = fis.read(buffer)) != -1) {
                digest.update(buffer, 0, read);
            }
            byte[] hash = digest.digest();
            StringBuilder sb = new StringBuilder();
            for (byte b : hash) {
                sb.append(String.format("%02x", b));
            }
            return sb.toString();
        } catch (java.security.NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    public static String readCachedHash(File cacheFile) {
        if (!cacheFile.exists()) return null;
        try (BufferedReader br = new BufferedReader(new FileReader(cacheFile))) {
            return br.readLine();
        } catch (IOException e) {
            return null;
        }
    }

    public static void writeCachedHash(File cacheFile, String hash) {
        try (PrintWriter pw = new PrintWriter(cacheFile)) {
            pw.println(hash);
        } catch (IOException e) {
            // log error but don't crash
            Main.LOGGER.log(Level.WARN, "Failed to write cached hash - " + hash);
        }
    }

}