package cc.polymorphism.obfuscator.mutator.impl.misc;

import cc.polymorphism.obfuscator.Polymorphism;
import cc.polymorphism.obfuscator.PolymorphismData;
import cc.polymorphism.obfuscator.asm.remapper.PolymorphismRemapper;
import cc.polymorphism.obfuscator.asm.wrapper.ClassWrapper;
import cc.polymorphism.obfuscator.asm.wrapper.FieldWrapper;
import cc.polymorphism.obfuscator.asm.wrapper.MethodWrapper;
import cc.polymorphism.obfuscator.dictionary.Dictionary;
import cc.polymorphism.obfuscator.dictionary.DictionaryFactory;
import cc.polymorphism.obfuscator.dictionary.defined.AlphabeticalDictionary;
import cc.polymorphism.obfuscator.logging.Logger;
import cc.polymorphism.obfuscator.mutator.Mutator;
import cc.polymorphism.obfuscator.util.RandomUtils;
import org.objectweb.asm.commons.ClassRemapper;
import org.objectweb.asm.tree.ClassNode;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.IntStream;

public class RenamingMutator extends Mutator {
    private String packagePrefix = "cc/polymorphism";

    private final List<String> adaptedResources = new ArrayList<>(List.of(
            "META-INF/MANIFEST.MF"
    ));

    private final Dictionary classNameDictionary = new AlphabeticalDictionary();
    private final Dictionary methodNameDictionary = new AlphabeticalDictionary();
    private final Dictionary fieldNameDictionary = new AlphabeticalDictionary();

    private final Map<String, String> mappings = new HashMap<>();

    public RenamingMutator(Polymorphism polymorphism) {
        super(polymorphism);
    }

    @Override
    public void transform() {
        Logger.info("Building inheritance hierarchy graph");
        long current = System.currentTimeMillis();
        Polymorphism.getInstance().buildHierarchyGraph();
        Logger.info(String.format("Finished building inheritance graph [%dms]", (System.currentTimeMillis() - current)));

        Logger.info("Generating mappings");
        current = System.currentTimeMillis();
        generateMappings();
        Logger.info(String.format("Finished generating mappings [%dms]", (System.currentTimeMillis() - current)));

        Logger.info("Applying mappings");
        current = System.currentTimeMillis();
        applyMappings();
        Logger.info(String.format("Finished applying mappings [%dms]", (System.currentTimeMillis() - current)));

        Logger.info("Adapting resources");
        current = System.currentTimeMillis();
        adaptResources();
        Logger.info(String.format("Finished adapting resources [%dms]", (System.currentTimeMillis() - current)));

        Logger.info("Dumping mappings");
        dumpMapping();

        classes().forEach(classWrapper -> {
            final ClassNode classNode = classWrapper.getClassNode();

            classNode.sourceFile = null;
            classNode.sourceDebug = null;

            classNode.methods.forEach(methodNode -> {
                methodNode.localVariables = null;
            });

            classNode.innerClasses.clear();

            classNode.outerClass = null;

            classNode.outerMethod = null;
            classNode.outerMethodDesc = null;
        });
    }

    private void generateMappings() {
        classes().forEach(classWrapper -> {
            classWrapper.methodStream().filter(methodWrapped -> !cannotRenameMethod(classWrapper, methodWrapped, new HashSet<>())).forEach(methodWrapper -> {
                String newName;
                do {
                    newName = methodNameDictionary.next();
                } while (mappings.containsKey(classWrapper.getOriginalName() + '.' + methodWrapper.getOriginalName() + methodWrapper.getOriginalDescriptor())
                        && mappings.get(classWrapper.getOriginalName() + '.' + methodWrapper.getOriginalName() + methodWrapper.getOriginalDescriptor()).equals(newName));


                generateMethodMappings(classWrapper, methodWrapper, newName);
            });
            classWrapper.fieldStream().filter(fieldWrapper -> !cannotRenameField(classWrapper, fieldWrapper)).forEach(fieldWrapper -> {
                String newName;
                do {
                    newName = fieldNameDictionary.next();
                } while (mappings.containsKey(classWrapper.getOriginalName() + '.' + fieldWrapper.getOriginalName() + ' ' + fieldWrapper.getOriginalType())
                        && mappings.get(classWrapper.getOriginalName() + '.' + fieldWrapper.getOriginalName() + ' ' + fieldWrapper.getOriginalType()).equals(newName));


                generateFieldMappings(classWrapper, fieldWrapper, newName);
            });

            if (packagePrefix == null) {
                packagePrefix = classNameDictionary.copy().randomStr(RandomUtils.randomInt(0xF));
            }

            // TODO: Exclusion
            var newName = packagePrefix;
            if (!newName.isEmpty()) {
                newName += '/';
            }
            String temp;
            do {
                temp = newName + classNameDictionary.next();
            } while (classPathMap().containsKey(temp)); // Important to check classpath instead of input classes
            newName = temp;

            mappings.put(classWrapper.getOriginalName(), newName);
        });
    }

    private boolean cannotRenameMethod(ClassWrapper classWrapper, MethodWrapper wrapper, Set<ClassWrapper> visited) {
        if (!visited.add(classWrapper)) {
            return false;
        }

        // TODO: Exclusion
        if (/* !notExcluded(classWrapper.getOriginalName() + '.' + wrapper.getOriginalName() + wrapper.getOriginalDescriptor())
                || */ mappings.containsKey(classWrapper.getOriginalName() + '.' + wrapper.getOriginalName() + wrapper.getOriginalDescriptor())) {
            return true;
        }

        if (wrapper.isNative()
                || wrapper.getOriginalName().equals("main")
                || wrapper.getOriginalName().equals("premain")
                || wrapper.getOriginalName().startsWith("<") ||
                (wrapper.getOriginalName().toLowerCase().contains("lambda")
                        && wrapper.getOriginalName().toLowerCase().contains("$")
                        || wrapper.getOwner().isAnnotation()
                        || wrapper.getOwner().isInterface())
        ) {
            return true;
        }

        if (wrapper.isStatic()) {
            return classWrapper.isEnum()
                    && (wrapper.getOriginalName().equals("valueOf") || wrapper.getOriginalName().equals("values"));
        } else {
            if (classWrapper != wrapper.getOwner() && classWrapper.isLibraryNode()
                    && classWrapper.methodStream().anyMatch(other -> other.getOriginalName().equals(wrapper.getOriginalName())
                    && other.getOriginalDescriptor().equals(wrapper.getOriginalDescriptor()))) {
                return true;
            }
            return classWrapper.getParents().stream().anyMatch(parent -> cannotRenameMethod(parent, wrapper, visited))
                    || classWrapper.getChildren().stream().anyMatch(child -> cannotRenameMethod(child, wrapper, visited));
        }
    }

    private void generateMethodMappings(ClassWrapper owner, MethodWrapper wrapper, String newName) {
        String key = owner.getOriginalName() + '.' + wrapper.getOriginalName() + wrapper.getOriginalDescriptor();

        if (mappings.containsKey(key)) {
            return;
        }
        mappings.put(key, newName);

        if (!wrapper.isStatic()) {
            owner.getParents().forEach(parent -> generateMethodMappings(parent, wrapper, newName));
            owner.getChildren().forEach(child -> generateMethodMappings(child, wrapper, newName));
        }
    }

    private boolean cannotRenameField(ClassWrapper classWrapper, FieldWrapper wrapper) {
        // TODO: Exclusion

        if (mappings.containsKey(classWrapper.getOriginalName() + '.' + wrapper.getOriginalName() + ' ' + wrapper.getOriginalType())) {
            return true;
        }

        return classWrapper.isEnum(); // Todo: enums are a pain to handle
    }

    private void generateFieldMappings(ClassWrapper owner, FieldWrapper wrapper, String newName) {
        String key = owner.getOriginalName() + '.' + wrapper.getOriginalName() + ' ' + wrapper.getOriginalType();

        if (mappings.containsKey(key)) {
            return;
        }
        mappings.put(key, newName);

        if (!wrapper.isStatic()) { //  Static fields cannot be inherited
            owner.getParents().forEach(parent -> generateFieldMappings(parent, wrapper, newName));
            owner.getChildren().forEach(child -> generateFieldMappings(child, wrapper, newName));
        }
    }

    private void applyMappings() {
        var remapper = new PolymorphismRemapper(mappings);
        new ArrayList<>(classes()).forEach(classWrapper -> {
            var classNode = classWrapper.getClassNode();
            var copy = new ClassNode();
            classNode.accept(new ClassRemapper(copy, remapper));

            IntStream.range(0, copy.methods.size()).forEach(i -> {
                classWrapper.getMethods().get(i).setMethodNode(copy.methods.get(i));
            });
            IntStream.range(0, copy.fields.size()).forEach(i -> {
                classWrapper.getFields().get(i).setFieldNode(copy.fields.get(i));
            });
            classWrapper.setClassNode(copy);

            classMap().remove(classWrapper.getOriginalName());
            classPathMap().remove(classWrapper.getOriginalName());
            classMap().put(copy.name, classWrapper);
            classPathMap().put(copy.name, classWrapper);
        });
    }

    private void adaptResources() {
        adaptedResources.forEach(resourceName -> {
            var bytes = resourceMap().get(resourceName);

            if (bytes == null) {
                Logger.warn("Attempted to adapt nonexistent resource: " + resourceName);
            }

            assert bytes != null : "Resource does not exist?";
            var stringVer = new String(bytes, StandardCharsets.UTF_8);

            for (var original : mappings.keySet()) {
                if (stringVer.contains(original.replace("/", "."))) {
                    if (resourceName.equals("META-INF/MANIFEST.MF")
                            || resourceName.equals("plugin.yml")
                            || resourceName.equals("bungee.yml")) {
                        stringVer = stringVer.replaceAll("(?<=[: ])" + original.replace("/", "."), mappings.get(original)).replace("/", ".");
                    } else {
                        stringVer = stringVer.replace(original.replace("/", "."), mappings.get(original)).replace("/", ".");
                    }
                }
            }

            resourceMap().put(resourceName, stringVer.getBytes(StandardCharsets.UTF_8));
        });
    }

    private void dumpMapping() {
        var file = new File("mappings.txt");

        try {
            file.createNewFile();

            var writer = new BufferedWriter(new FileWriter(file));

            mappings.forEach((oldName, newName) -> {
                try {
                    writer.append(oldName).append(" -> ").append(newName).append("\n");
                } catch (IOException ioe) {
                    Logger.warn(String.format("Caught IOException while attempting to write line \"%s -> %s\"", oldName, newName));
                    if (PolymorphismData.VERBOSE) {
                        ioe.printStackTrace(System.out);
                    }
                }
            });
        } catch (Throwable throwable) {
            Logger.warn("Captured throwable upon attempting to generate mappings file: " + throwable.getMessage());

            if (PolymorphismData.VERBOSE) {
                throwable.printStackTrace(System.out);
            }
        }
    }

    @Override
    public String getConfigName() {
        return "renaming";
    }
}
