package cc.polymorphism.obfuscator.mutator;

import cc.polymorphism.obfuscator.Polymorphism;
import cc.polymorphism.obfuscator.asm.wrapper.ClassWrapper;
import cc.polymorphism.obfuscator.cli.config.setting.Setting;
import cc.polymorphism.obfuscator.cli.config.setting.impl.BooleanSetting;
import cc.polymorphism.obfuscator.cli.config.setting.impl.MultiStringSetting;
import cc.polymorphism.obfuscator.cli.config.setting.impl.StringSetting;
import cc.polymorphism.obfuscator.engine.hash.HashGenerator;
import cc.polymorphism.obfuscator.engine.seed.SeedGenerator;
import cc.polymorphism.obfuscator.util.RandomUtils;
import lombok.Getter;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.tree.ClassNode;

import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Stream;

@SuppressWarnings("unused")
public abstract class Mutator implements Opcodes {
    private final Polymorphism ctx;

    private final BooleanSetting state = new BooleanSetting("enabled", false);
    private final MultiStringSetting exclusion = new MultiStringSetting("exclusions", List.of());

    @Getter
    private final List<Setting<?>> settings = new ArrayList<>();

    public Mutator(Polymorphism ctx) {
        this.ctx = ctx;
        addSetting(state, exclusion);
    }

    public void addSetting(Setting<?>... settings) {
        this.settings.addAll(Arrays.asList(settings));
    }

    public Setting<?> getSetting(String name) {
        return this.settings.stream()
                .filter(setting -> setting.getName().equals(name))
                .findAny().orElseThrow(() -> new IllegalStateException(String.format("Setting \"%s\" was not found!", name.toUpperCase())));
    }

    public boolean enabled() {
        return this.state.getValue();
    }

    public HashGenerator hashGenerator() {
        return this.ctx.getHashGenerator();
    }

    public SeedGenerator seedGenerator() {
        return this.ctx.getSeedGenerator();
    }

    protected Collection<ClassWrapper> classes() {
        return ctx.getClasses().values();
    }

    protected Stream<ClassWrapper> classStream() {
        return classes().stream();
    }

    protected Map<String, ClassWrapper> classMap() {
        return ctx.getClasses();
    }

    protected Map<String, ClassWrapper> classPathMap() {
        return ctx.getClasspath();
    }

    protected Map<String, byte[]> resourceMap() {
        return ctx.getResources();
    }

    protected void addClass(ClassWrapper classWrapper) {
        ctx.getClasses().put(classWrapper.getName(), classWrapper);
        ctx.getClasspath().put(classWrapper.getName(), classWrapper);
    }

    protected void addClass(ClassNode classNode) {
        addClass(new ClassWrapper(classNode, false));
    }

    protected ClassWrapper randomClass() {
        var list = new ArrayList<>(classMap().values());
        return list.get(RandomUtils.randomInt(list.size()));
    }

    public String getName() {
        return getClass().getName();
    }

    public boolean isExcluded(ClassWrapper classWrapper) {
        boolean excluded = false;

        for (final String regex : exclusion.getValue()) {
            if (Pattern.matches(regex, classWrapper.getOriginalName())) {
                excluded = true;
            }
        }

        return excluded;
    }

    public boolean isIncluded(ClassWrapper classWrapper) {
        return !isExcluded(classWrapper);
    }

    public abstract void transform();

    public abstract String getConfigName();
}
