package cc.polymorphism.obfuscator.engine.seed;

import cc.polymorphism.assembly.BytecodeBlock;
import cc.polymorphism.assembly.expressions.IRVariable;
import cc.polymorphism.obfuscator.asm.wrapper.ClassWrapper;
import cc.polymorphism.obfuscator.asm.wrapper.MethodWrapper;
import org.objectweb.asm.tree.ClassNode;
import org.objectweb.asm.tree.FieldNode;
import org.objectweb.asm.tree.MethodNode;
import org.objectweb.asm.tree.TryCatchBlockNode;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class SeedBuilder {
    private ClassWrapper classWrapper;
    private MethodWrapper methodWrapper;

    private IRVariable variable;
    private int value;

    private final List<ClassWrapper> classes = new ArrayList<>();

    private final Map<ClassNode, FieldNode> fields = new HashMap<>();
    private final Map<ClassNode, MethodNode> methods = new HashMap<>();

    private final Map<MethodNode, TryCatchBlockNode> traps = new HashMap<>();

    private BytecodeBlock block;

    public SeedBuilder setClassWrapper(ClassWrapper classWrapper) {
        this.classWrapper = classWrapper;
        return this;
    }

    public SeedBuilder setMethodWrapper(MethodWrapper methodWrapper) {
        this.methodWrapper = methodWrapper;
        return this;
    }

    public SeedBuilder setVariable(IRVariable variable) {
        this.variable = variable;
        return this;
    }

    public SeedBuilder setValue(int value) {
        this.value = value;
        return this;
    }

    public SeedBuilder addClass(ClassWrapper classWrapper) {
        this.classes.add(classWrapper);
        return this;
    }

    public SeedBuilder addField(ClassNode classNode, FieldNode fieldNode) {
        this.fields.put(classNode, fieldNode);
        return this;
    }

    public SeedBuilder addMethod(ClassNode classNode, MethodNode methodNode) {
        this.methods.put(classNode, methodNode);
        return this;
    }

    public SeedBuilder addTrap(MethodNode methodNode, TryCatchBlockNode trap) {
        this.traps.put(methodNode, trap);
        return this;
    }

    public SeedBuilder setBlock(BytecodeBlock block) {
        this.block = block;
        return this;
    }

    public Seed build() {
        if (classWrapper == null)
            throw new IllegalStateException("Invalid seed: Klass not set.");

        if (methodWrapper == null)
            throw new IllegalStateException("Invalid seed: Method not set.");

        if (variable == null)
            throw new IllegalStateException("Invalid seed: Variable not set.");

        if (block == null)
            throw new IllegalStateException("Invalid seed: Block not set.");

        return new Seed(classWrapper, methodWrapper, variable, value, classes, fields, methods, traps, block);
    }
}