package cc.polymorphism.obfuscator.cli.config;

import cc.polymorphism.obfuscator.Polymorphism;
import cc.polymorphism.obfuscator.cli.config.setting.Setting;
import cc.polymorphism.obfuscator.cli.config.setting.impl.BooleanSetting;
import cc.polymorphism.obfuscator.cli.config.setting.impl.MultiStringSetting;
import cc.polymorphism.obfuscator.cli.config.setting.impl.NumberSetting;
import cc.polymorphism.obfuscator.cli.config.setting.impl.StringSetting;
import cc.polymorphism.obfuscator.logging.Logger;
import cc.polymorphism.obfuscator.mutator.Mutator;
import cc.polymorphism.obfuscator.util.FileUtils;
import com.google.gson.*;

import java.io.File;
import java.util.List;

public class ConfigHandler {
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();

    public static Config read(final Polymorphism ctx, final File file) {
        final var json = FileUtils.readJsonFromFile(GSON, file);

        if (json == null) {
            System.err.println("Failed to parse configuration file!");
            System.exit(0);
            return null;
        }

        final Config config = new Config();

        config.setInput(json.get("input").getAsString());
        config.setOutput(json.get("output").getAsString());

        final List<String> dependencies = json.getAsJsonArray("libraries").asList().stream()
                .map(JsonElement::getAsString)
                .toList();

        config.setDependencies(dependencies);

        final var transformerArray = json.getAsJsonArray("transformers");
        for (int i = 0; i < transformerArray.size(); i++) {
            final var transformerObj = transformerArray.get(i).getAsJsonObject();

            final var transformerName = transformerObj.get("name").getAsString();
            final var transformer = ctx.getMutatorManager()
                    .getBy(o -> o.getConfigName().equals(transformerName));

            for (Setting<?> setting : transformer.getSettings()) {
                final JsonElement element = transformerObj.get(setting.getName());

                try {
                    switch (setting) {
                        case BooleanSetting booleanSetting -> booleanSetting.setValue(element.getAsBoolean());
                        case StringSetting stringSetting -> stringSetting.setValue(element.getAsString());
                        case NumberSetting numberSetting -> numberSetting.setValue(element.getAsFloat());
                        case MultiStringSetting multiStringSetting -> {
                            final List<String> strings = element.getAsJsonArray().asList().stream()
                                    .map(JsonElement::getAsString)
                                    .toList();

                            multiStringSetting.setValue(strings);
                        }
                        default -> {
                            /* */
                        }
                    }
                } catch (Exception e) {
                    Logger.warn("Could not find a setting!");
                }
            }
        }

        return config;
    }

    public static void write(final Polymorphism ctx, final File file) {
        final var obj = new JsonObject();

        obj.addProperty("input", "input.jar");
        obj.addProperty("output", "output.jar");
        obj.add("libraries", new JsonArray());

        // Array all transformers will be in
        final var transformersArr = new JsonArray();

        // Add transformer stuff to transformers array
        for (Mutator mutator : ctx.getMutatorManager().getObjects()) {
            final var transformerObj = serializeTransformer(mutator);
            transformersArr.add(transformerObj);
        }

        obj.add("transformers", transformersArr);

        FileUtils.writeJsonToFile(GSON, obj, file);
    }

    private static JsonObject serializeTransformer(Mutator mutator) {
        final var transformerObj = new JsonObject();
        transformerObj.addProperty("name", mutator.getConfigName());

        for (Setting<?> setting : mutator.getSettings()) {
            if (setting instanceof BooleanSetting booleanSetting) {
                transformerObj.addProperty(setting.getName(), booleanSetting.getValue());
            } else if (setting instanceof StringSetting stringSetting)
                transformerObj.addProperty(setting.getName(), stringSetting.getValue());
            else if (setting instanceof NumberSetting numberSetting)
                transformerObj.addProperty(setting.getName(), numberSetting.getValue());
            else if (setting instanceof MultiStringSetting multiStringSetting) {
                final JsonArray array = new JsonArray();

                for (String string : multiStringSetting.getValue()) {
                    array.add(string);
                }

                transformerObj.add(setting.getName(), array);
            }
        }
        return transformerObj;
    }
}
