package cc.polymorphism.obfuscator.cli;

import cc.polymorphism.obfuscator.exceptions.obfuscation.PreventableObfuscationException;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

public class CmdLineParser {
    private final static String[] SWITCH_PREFIXES = {"--", "-", "/"};
    private final static Set<CmdSwitch> SWITCHES = new HashSet<>();
    private final Map<String, String[]> argMap = new HashMap<>();

    private String getSwitchIdOrThrow(String arg) {
        return arg.substring(Stream.of(SWITCH_PREFIXES)
                .filter(arg::startsWith)
                .findFirst()
                .orElseThrow(() -> new PreventableObfuscationException("Unexpected command argument: \"" + arg + "\""))
                .length());
    }

    public void parse(String[] args) {
        for (var cliArgIndex = 0; cliArgIndex < args.length; cliArgIndex++) {
            final var switchId = getSwitchIdOrThrow(args[cliArgIndex]);

            var knownSwitch = false;
            for (var cmdSwitch : SWITCHES) {
                if (cmdSwitch.id.equals(switchId)) {
                    String[] switchArgs = new String[cmdSwitch.expectedArgs];

                    for (var switchArgIndex = 0; switchArgIndex < cmdSwitch.expectedArgs; switchArgIndex++) {
                        try {
                            switchArgs[switchArgIndex] = args[++cliArgIndex];
                        } catch (ArrayIndexOutOfBoundsException e) {
                            throw new PreventableObfuscationException(String.format("Command switch \"%s\" expected %d argument(s), got %d instead.",
                                    switchId,
                                    cmdSwitch.expectedArgs,
                                    switchArgIndex
                            ));
                        }
                    }

                    argMap.put(switchId, switchArgs);
                    knownSwitch = true;
                    break;
                }
            }

            if (!knownSwitch) {
                throw new PreventableObfuscationException("Unknown command switch: \"" + switchId + "\"");
            }
        }
    }

    public static void registerSwitch(String id, int expectedArgs) {
        registerSwitch(new CmdSwitch(id, expectedArgs));
    }

    public static void registerSwitch(CmdSwitch cmdSwitch) {
        SWITCHES.add(cmdSwitch);
    }

    public boolean containsSwitch(String id) {
        return argMap.containsKey(id);
    }

    public String[] getArgsFor(String id) {
        return argMap.get(id);
    }
}
