package cc.polymorphism.obfuscator.asm.wrapper;

import cc.polymorphism.obfuscator.analysis.LocalVariableAnalyser;
import cc.polymorphism.obfuscator.analysis.LocalVariableObject;
import cc.polymorphism.obfuscator.engine.seed.Seed;
import cc.polymorphism.obfuscator.util.ASMUtils;
import cc.polymorphism.obfuscator.util.RandomUtils;
import lombok.Getter;
import lombok.Setter;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.commons.CodeSizeEvaluator;
import org.objectweb.asm.tree.*;
import org.objectweb.asm.tree.analysis.Analyzer;
import org.objectweb.asm.tree.analysis.Frame;
import org.objectweb.asm.tree.analysis.SourceInterpreter;
import org.objectweb.asm.tree.analysis.SourceValue;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Setter
@Getter
public class MethodWrapper implements Opcodes {
    private static final int MAX_CODE_SIZE = 0xFFFF;

    private MethodNode methodNode;
    private final String originalName;
    private final String originalDescriptor;
    private final ClassWrapper owner;

    private final List<LocalVariableObject> locals = new ArrayList<>();

    private final List<Seed> seeds = new ArrayList<>();

    public MethodWrapper(MethodNode methodNode, ClassWrapper owner) {
        this.methodNode = methodNode;
        this.originalName = methodNode.name;
        this.originalDescriptor = methodNode.desc;
        this.owner = owner;

        LocalVariableAnalyser.computeLocalVariables(this);
    }

    public String getName() {
        return methodNode.name;
    }

    public String getDescriptor() {
        return methodNode.desc;
    }

    // ------------
    // Computation
    // ------------

    public void preComputeLocals() {
        LocalVariableAnalyser.computeLocalVariables(this);

        final var methodInit = new InsnList();

        for (LocalVariableObject local : this.locals) {
            switch (local.type()) {
                case INT -> {
                    methodInit.add(ASMUtils.getNumberInsn(RandomUtils.randomInt()));
                    methodInit.add(new VarInsnNode(ISTORE, local.idx()));
                }
                case LONG -> {
                    if (RandomUtils.randomBoolean()) {
                        methodInit.add(ASMUtils.getNumberInsn(RandomUtils.randomInt()));
                        methodInit.add(new InsnNode(I2L));
                        methodInit.add(new VarInsnNode(LSTORE, local.idx()));
                    } else {
                        methodInit.add(new LdcInsnNode(RandomUtils.randomLong()));
                        methodInit.add(new VarInsnNode(LSTORE, local.idx()));
                    }
                }
                case DOUBLE -> {
                    methodInit.add(new LdcInsnNode(RandomUtils.randomDouble()));
                    methodInit.add(new VarInsnNode(DSTORE, local.idx()));
                }
                case FLOAT -> {
                    methodInit.add(new LdcInsnNode(RandomUtils.randomFloat()));
                    methodInit.add(new VarInsnNode(FSTORE, local.idx()));
                }
                case OBJECT -> {
                    methodInit.add(new InsnNode(ACONST_NULL));
                    methodInit.add(new VarInsnNode(ASTORE, local.idx()));
                }
            }
        }

        methodNode.instructions.insertBefore(methodNode.instructions.getFirst(), methodInit);
    }

    public Map<AbstractInsnNode, Frame<SourceValue>> analyzeSource() {
        this.methodNode.maxStack = Byte.MAX_VALUE;
        try {
            final Map<AbstractInsnNode, Frame<SourceValue>> frames = new HashMap<>();

            final Frame<SourceValue>[] framesArray = new Analyzer<>(new SourceInterpreter()).
                    analyze(this.owner.getClassNode().name, this.methodNode);

            for (int i = 0; i < framesArray.length; i++) {
                frames.put(this.methodNode.instructions.get(i), framesArray[i]);
            }
            return frames;
        } catch (Exception e) {
            return null;
        }
    }

    // ------------
    // Access stuff
    // ------------

    public void addAccessFlags(int flags) {
        methodNode.access |= flags;
    }

    public void removeAccessFlags(int flags) {
        methodNode.access &= ~flags;
    }

    public boolean isPublic() {
        return (ACC_PUBLIC & methodNode.access) != 0;
    }

    public boolean isPrivate() {
        return (ACC_PRIVATE & methodNode.access) != 0;
    }

    public boolean isProtected() {
        return (ACC_PROTECTED & methodNode.access) != 0;
    }

    public boolean isStatic() {
        return (ACC_STATIC & methodNode.access) != 0;
    }

    public boolean isFinal() {
        return (ACC_FINAL & methodNode.access) != 0;
    }

    public boolean isSynchronized() {
        return (ACC_SYNCHRONIZED & methodNode.access) != 0;
    }

    public boolean isBridge() {
        return (ACC_BRIDGE & methodNode.access) != 0;
    }

    public boolean isVarargs() {
        return (ACC_VARARGS & methodNode.access) != 0;
    }

    public boolean isNative() {
        return (ACC_NATIVE & methodNode.access) != 0;
    }

    public boolean isAbstract() {
        return (ACC_ABSTRACT & methodNode.access) != 0;
    }

    public boolean isStrict() {
        return (ACC_STRICT & methodNode.access) != 0;
    }

    public boolean isSynthetic() {
        return (ACC_SYNTHETIC & methodNode.access) != 0;
    }

    public boolean isMandated() {
        return (ACC_MANDATED & methodNode.access) != 0;
    }

    public boolean isDeprecated() {
        return (ACC_DEPRECATED & methodNode.access) != 0;
    }

    public boolean isSpecial() {
        return methodNode.name.startsWith("<");
    }

    // -----
    // Misc.
    // -----

    public boolean hasInstructions() {
        return methodNode.instructions.size() > 0;
    }

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    public boolean hasVisibleAnnotations() {
        return methodNode.visibleAnnotations != null && !methodNode.visibleAnnotations.isEmpty();
    }

    public int getCodeSize() {
        CodeSizeEvaluator evaluator = new CodeSizeEvaluator(null);
        methodNode.accept(evaluator);
        return evaluator.getMaxSize();
    }

    public int getLeewaySize() {
        return MAX_CODE_SIZE - getCodeSize();
    }

    public static MethodWrapper from(MethodNode methodNode, ClassWrapper owner) {
        return new MethodWrapper(methodNode, owner);
    }
}
