package cc.polymorphism.obfuscator.asm.wrapper;

import cc.polymorphism.obfuscator.PolymorphismData;
import cc.polymorphism.obfuscator.asm.PolymorphismClassWriter;
import cc.polymorphism.obfuscator.dictionary.Dictionary;
import cc.polymorphism.obfuscator.engine.hash.Hash;
import cc.polymorphism.obfuscator.logging.Logger;
import lombok.Getter;
import lombok.Setter;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassWriter;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.tree.ClassNode;
import org.objectweb.asm.tree.FieldNode;
import org.objectweb.asm.tree.MethodNode;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

/**
 * Wrapper around {@link ClassNode}.
 *
 * @author itzsomebody
 */
@Getter
@Setter
public class ClassWrapper implements Opcodes {
    private static final int LIB_READER_FLAGS = ClassReader.SKIP_FRAMES | ClassReader.SKIP_DEBUG | ClassReader.SKIP_CODE;
    private static final int INPUT_READER_FLAGS = ClassReader.SKIP_FRAMES | ClassReader.SKIP_DEBUG;

    private ClassNode classNode;
    private final String originalName;
    private final boolean libraryNode;

    private final List<MethodWrapper> methods;
    private final List<FieldWrapper> fields;

    private final List<ClassWrapper> parents = new ArrayList<>();
    private final List<ClassWrapper> children = new ArrayList<>();

    private Hash hash;

    public List<String> utf8Consts = new ArrayList<>() {
        {
            add("Polymorphism" + PolymorphismData.VERSION);
        }
    };

    public ClassWrapper(ClassReader reader, boolean libraryNode) {
        var classNode = new ClassNode();
        reader.accept(classNode, libraryNode ? LIB_READER_FLAGS : INPUT_READER_FLAGS);

        this.classNode = classNode;
        this.originalName = classNode.name;
        this.libraryNode = libraryNode;

        this.methods = MethodWrappers.from(this);
        this.fields = FieldWrappers.from(this);
    }

    public ClassWrapper(ClassNode classNode, boolean libraryNode) {
        this.classNode = classNode;
        this.originalName = classNode.name;
        this.libraryNode = libraryNode;

        this.methods = MethodWrappers.from(this);
        this.fields = FieldWrappers.from(this);
    }

    // -----------------
    // Getters / Setters
    // -----------------

    public String getName() {
        return classNode.name;
    }

    public String getNormalizedName() {
        return classNode.name.replace("/", ".");
    }

    public String getSuperName() {
        return classNode.superName;
    }

    public List<String> getInterfaceNames() {
        return classNode.interfaces;
    }

    public Stream<MethodWrapper> methodStream() {
        return getMethods().stream();
    }

    public Stream<FieldWrapper> fieldStream() {
        return getFields().stream();
    }

    // ------------
    // Access stuff
    // ------------

    public void addAccessFlags(int flags) {
        classNode.access |= flags;
    }

    public void removeAccessFlags(int flags) {
        classNode.access &= ~flags;
    }

    public boolean isPublic() {
        return (classNode.access & ACC_PUBLIC) != 0;
    }

    public boolean isPrivate() {
        return (classNode.access & ACC_PRIVATE) != 0;
    }

    public boolean isProtected() {
        return (classNode.access & ACC_PROTECTED) != 0;
    }

    public boolean isFinal() {
        return (classNode.access & ACC_FINAL) != 0;
    }

    public boolean isSuper() {
        return (classNode.access & ACC_SUPER) != 0;
    }

    public boolean isInterface() {
        return (classNode.access & ACC_INTERFACE) != 0;
    }

    public boolean isAbstract() {
        return (classNode.access & ACC_ABSTRACT) != 0;
    }

    public boolean isSynthetic() {
        return (classNode.access & ACC_SYNTHETIC) != 0;
    }

    public boolean isAnnotation() {
        return (classNode.access & ACC_ANNOTATION) != 0;
    }

    public boolean isEnum() {
        return (classNode.access & ACC_ENUM) != 0;
    }

    public boolean isModule() {
        return (classNode.access & ACC_MODULE) != 0;
    }

    public boolean isRecord() {
        return (classNode.access & ACC_RECORD) != 0;
    }

    public boolean isDeprecated() {
        return (classNode.access & ACC_DEPRECATED) != 0;
    }

    // -----
    // Misc.
    // -----

    public void addMethod(MethodNode methodNode) {
        classNode.methods.add(methodNode);
        methods.add(MethodWrapper.from(methodNode, this));
    }

    public void addField(FieldNode fieldNode) {
        classNode.fields.add(fieldNode);
        fields.add(FieldWrapper.from(fieldNode, this));
    }

    public MethodNode getMethodNode(String name, String desc) {
        return classNode.methods.stream().filter(methodNode -> name.equals(methodNode.name) && desc.equals(methodNode.desc)).findAny().orElse(null);
    }

    public FieldNode getFieldNode(String name, String desc) {
        return classNode.fields.stream().filter(fieldNode -> name.equals(fieldNode.name) && desc.equals(fieldNode.desc)).findAny().orElse(null);
    }

    public boolean containsMethodNode(String name, String desc) {
        return classNode.methods.stream().anyMatch(methodNode -> name.equals(methodNode.name) && desc.equals(methodNode.desc));
    }

    public boolean containsFieldNode(String name, String desc) {
        return classNode.fields.stream().anyMatch(fieldNode -> name.equals(fieldNode.name) && desc.equals(fieldNode.desc));
    }

    public String generateNextAllowedMethodName(Dictionary dictionary, String desc) {
        String name = dictionary.next();
        while (containsMethodNode(name, desc)) {
            name = dictionary.next();
        }
        return name;
    }

    public String generateNextAllowedFieldName(Dictionary dictionary, String desc) {
        String name = dictionary.next();
        while (containsFieldNode(name, desc)) {
            name = dictionary.next();
        }
        return name;
    }

    public boolean allowsConstDy() {
        return (classNode.version >= V11) && (classNode.version != V1_1);
    }

    public boolean isInvokeDynamicAllowed() {
        return (classNode.version >= V1_7) && (classNode.version != V1_1);
    }

    public boolean allowsJsr() {
        return (classNode.version <= V1_5) || (classNode.version == V1_1);
    }

    public boolean hasVisibleAnnotations() {
        return classNode.visibleAnnotations != null && !classNode.visibleAnnotations.isEmpty();
    }

    public byte[] toByteArray() {
        ClassWriter classWriter = new PolymorphismClassWriter(ClassWriter.COMPUTE_FRAMES);

        try {
            // Write all non-essential (manually made) entries first
            utf8Consts.forEach(classWriter::newUTF8);

            // Do the rest
            classNode.accept(classWriter);
            return classWriter.toByteArray();
        } catch (Throwable t) {
            Logger.warn(String.format("Error writing class %s. Skipping frames (might cause runtime errors).", classNode.name + ".class"));
            t.printStackTrace(System.out);
            classWriter = new ClassWriter(ClassWriter.COMPUTE_MAXS);
            utf8Consts.forEach(classWriter::newUTF8);
            classNode.accept(classWriter);
            return classWriter.toByteArray();
        }
    }

    public static ClassWrapper from(ClassReader reader) {
        return new ClassWrapper(reader, false);
    }

    public static ClassWrapper fromLib(ClassReader reader) {
        return new ClassWrapper(reader, true);
    }
}
