package cc.polymorphism.obfuscator.asm;

import lombok.Getter;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.tree.*;

import java.util.HashSet;
import java.util.Set;

@SuppressWarnings("unused")
public class StackEmulator implements Opcodes {

    private final MethodNode methodNode;
    private final AbstractInsnNode breakPoint;

    @Getter
    private final Set<AbstractInsnNode> emptyStack;

    public StackEmulator(MethodNode methodNode, AbstractInsnNode breakPoint) {
        this.methodNode = methodNode;
        this.breakPoint = breakPoint;
        this.emptyStack = new HashSet<>();
    }

    @SuppressWarnings("DataFlowIssue")
    public void emulate(boolean debug) throws RuntimeException {
        int stackSize = 0;
        Set<LabelNode> excHandlers = new HashSet<>();

        methodNode.tryCatchBlocks.forEach(tryCatchBlockNode -> excHandlers.add(tryCatchBlockNode.handler));

        for (int i = 0; i < this.methodNode.instructions.size(); i++) {
            AbstractInsnNode ain = this.methodNode.instructions.get(i);

            if (ain instanceof LabelNode && excHandlers.contains(ain))
                stackSize = 1;

            if (stackSize < 0)
                throw new RuntimeException("Stack height is negative!");
            if (stackSize == 0)
                this.emptyStack.add(ain);

            if (this.breakPoint == ain)
                break;

            switch (ain.getOpcode()) {
                case ACONST_NULL:
                case ICONST_M1:
                case ICONST_0:
                case ICONST_1:
                case ICONST_2:
                case ICONST_3:
                case ICONST_4:
                case ICONST_5:
                case FCONST_0:
                case FCONST_1:
                case FCONST_2:
                case BIPUSH:
                case SIPUSH:
                case ILOAD:
                case FLOAD:
                case ALOAD:
                case DUP:
                case DUP_X1:
                case DUP_X2:
                case I2L:
                case I2D:
                case F2L:
                case F2D:
                case NEW:
                    stackSize++;
                    break;
                case LDC:
                    LdcInsnNode ldc = (LdcInsnNode) ain;
                    if (ldc.cst instanceof Long || ldc.cst instanceof Double)
                        stackSize++;

                    stackSize++;
                    break;
                case LCONST_0:
                case LCONST_1:
                case DCONST_0:
                case DCONST_1:
                case LLOAD:
                case DLOAD:
                case DUP2:
                case DUP2_X1:
                case DUP2_X2:
                    stackSize += 2;
                    break;
                case IALOAD:
                case FALOAD:
                case AALOAD:
                case BALOAD:
                case CALOAD:
                case SALOAD:
                case ISTORE:
                case FSTORE:
                case ASTORE:
                case POP:
                case IADD:
                case FADD:
                case ISUB:
                case FSUB:
                case IMUL:
                case FMUL:
                case IDIV:
                case FDIV:
                case IREM:
                case FREM:
                case ISHL:
                case ISHR:
                case IUSHR:
                case LSHL:
                case LSHR:
                case LUSHR:
                case IAND:
                case IOR:
                case IXOR:
                case L2I:
                case L2F:
                case D2I:
                case D2F:
                case FCMPL:
                case FCMPG:
                case IFEQ:
                case IFNE:
                case IFLT:
                case IFGE:
                case IFGT:
                case IFLE:
                case TABLESWITCH:
                case LOOKUPSWITCH:
                case IRETURN:
                case FRETURN:
                case ATHROW:
                case MONITORENTER:
                case MONITOREXIT:
                case IFNULL:
                case IFNONNULL:
                case ARETURN:
                    stackSize--;
                    break;
                case LSTORE:
                case DSTORE:
                case POP2:
                case LADD:
                case DADD:
                case LSUB:
                case DSUB:
                case LMUL:
                case DMUL:
                case LDIV:
                case DDIV:
                case LREM:
                case DREM:
                case LAND:
                case LOR:
                case LXOR:
                case IF_ICMPEQ:
                case IF_ICMPNE:
                case IF_ICMPLT:
                case IF_ICMPGE:
                case IF_ICMPGT:
                case IF_ICMPLE:
                case IF_ACMPEQ:
                case IF_ACMPNE:
                case LRETURN:
                case DRETURN:
                    stackSize -= 2;
                    break;
                case IASTORE:
                case FASTORE:
                case AASTORE:
                case BASTORE:
                case CASTORE:
                case SASTORE:
                case LCMP:
                case DCMPL:
                case DCMPG:
                    stackSize -= 3;
                    break;
                case LASTORE:
                case DASTORE:
                    stackSize -= 4;
                    break;
                case GETSTATIC:
                    stackSize += doFieldEmulation(((FieldInsnNode) ain).desc, true);
                    break;
                case PUTSTATIC:
                    stackSize += doFieldEmulation(((FieldInsnNode) ain).desc, false);
                    break;
                case GETFIELD:
                    stackSize--;
                    stackSize += doFieldEmulation(((FieldInsnNode) ain).desc, true);
                    break;
                case PUTFIELD:
                    stackSize--;
                    stackSize += doFieldEmulation(((FieldInsnNode) ain).desc, false);
                    break;
                case INVOKEVIRTUAL:
                case INVOKESPECIAL:
                case INVOKEINTERFACE:
                    stackSize--;
                    stackSize += doMethodEmulation(((MethodInsnNode) ain).desc);
                    break;
                case INVOKESTATIC:
                    stackSize += doMethodEmulation(((MethodInsnNode) ain).desc);
                    break;
                case INVOKEDYNAMIC:
                    stackSize += doMethodEmulation(((InvokeDynamicInsnNode) ain).desc);
                    break;
                case MULTIANEWARRAY:
                    stackSize -= ((MultiANewArrayInsnNode) ain).dims;
                    stackSize++;
                    break;
                case JSR:
                case RET:
                    throw new RuntimeException("Did not expect JSR/RET instructions!");
                default:
                    break;
            }
        }
    }

    private static int doFieldEmulation(String desc, boolean isGet) {
        Type type = Type.getType(desc);
        int result = (type.getSort() == Type.LONG || type.getSort() == Type.DOUBLE) ? 2 : 1;

        return (isGet) ? result : -result;
    }

    private static int doMethodEmulation(String desc) {
        int result = 0;
        Type[] args = Type.getArgumentTypes(desc);
        Type returnType = Type.getReturnType(desc);
        for (Type type : args) {
            if (type.getSort() == Type.LONG || type.getSort() == Type.DOUBLE)
                result--;

            result--;
        }
        if (returnType.getSort() == Type.LONG || returnType.getSort() == Type.DOUBLE)
            result++;
        if (returnType.getSort() != Type.VOID)
            result++;

        return result;
    }
}