package cc.polymorphism.obfuscator.asm;

import cc.polymorphism.obfuscator.Polymorphism;
import cc.polymorphism.obfuscator.asm.wrapper.ClassWrapper;
import org.objectweb.asm.ClassWriter;

import java.util.ArrayDeque;
import java.util.HashSet;

/**
 * Custom-implemented version of {@link ClassWriter} which doesn't use the internal JVM classpath when computing
 * stackmap frames.
 * <p>
 * TODO: Perhaps consider creating fully-worked out hierarchies for JVM runtime classes so we don't have to keep checking huge jars
 * TODO: Implement shadow classes
 *
 * @author itzsomebody
 */
public class PolymorphismClassWriter extends ClassWriter {
    private static final Polymorphism POLYMORPHISM = Polymorphism.getInstance();

    public PolymorphismClassWriter(int flags) {
        super(flags);
    }

    @Override
    protected String getCommonSuperClass(final String type1, final String type2) {
        if ("java/lang/Object".equals(type1) || "java/lang/Object".equals(type2))
            return "java/lang/Object";

        String first = deriveCommonSuperName(type1, type2);
        String second = deriveCommonSuperName(type2, type1);
        if (!"java/lang/Object".equals(first)) {
            return first;
        }
        if (!"java/lang/Object".equals(second)) {
            return second;
        }

        return getCommonSuperClass(POLYMORPHISM.getClasspathWrapper(type1).getSuperName(), POLYMORPHISM.getClasspathWrapper(type2).getSuperName());
    }

    private String deriveCommonSuperName(final String type1, final String type2) {
        ClassWrapper first = POLYMORPHISM.getClasspathWrapper(type1);
        ClassWrapper second = POLYMORPHISM.getClasspathWrapper(type2);
        if (isAssignableFrom(type1, type2)) {
            return type1;
        } else if (isAssignableFrom(type2, type1)) {
            return type2;
        } else if (first.isInterface() || second.isInterface()) {
            return "java/lang/Object";
        } else {
            String temp;

            do {
                temp = first.getSuperName();
                first = POLYMORPHISM.getClasspathWrapper(temp);
            } while (!isAssignableFrom(temp, type2));
            return temp;
        }
    }

    private boolean isAssignableFrom(String type1, String type2) {
        if ("java/lang/Object".equals(type1)) {
            return true;
        }
        if (type1.equals(type2)) {
            return true;
        }

        ClassWrapper first = POLYMORPHISM.getClasspathWrapper(type1);
        POLYMORPHISM.getClasspathWrapper(type2);

        var allChildren = new HashSet<String>();
        var toProcess = new ArrayDeque<String>();
        first.getChildren().forEach(child -> {
            toProcess.add(child.getName());
        });

        while (!toProcess.isEmpty()) {
            String next = toProcess.poll();

            if (allChildren.add(next)) {
                ClassWrapper temp = POLYMORPHISM.getClasspathWrapper(next);
                temp.getChildren().forEach(child -> {
                    toProcess.add(child.getName());
                });
            }
        }
        return allChildren.contains(type2);
    }
}
