package cc.polymorphism.obfuscator.asm;

import cc.polymorphism.obfuscator.PolymorphismData;
import cc.polymorphism.obfuscator.asm.wrapper.ClassWrapper;
import cc.polymorphism.obfuscator.exceptions.obfuscation.PreventableObfuscationException;
import cc.polymorphism.obfuscator.logging.Logger;
import cc.polymorphism.obfuscator.util.IOUtils;
import lombok.Getter;
import org.objectweb.asm.ClassReader;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

@Getter
public class JarLoader {
    private final Map<String, ClassWrapper> classes = new HashMap<>();
    private final Map<String, ClassWrapper> classpath = new HashMap<>();
    private final Map<String, byte[]> resources = new HashMap<>();

    public void loadAsLib(String path) {
        Logger.info("Loading library \"" + path + "\"");
        var file = new File(path);

        if (!file.exists()) {
            Logger.warn("Library \"" + file.getAbsolutePath() + "\" doesn't exist. Skipping");
        }
        if (!file.canRead()) {
            Logger.warn("Library \"" + file.getAbsolutePath() + "\" doesn't have read permissions. Skipping");
        }
        if (!file.isFile()) {
            Logger.warn("Library \"" + file.getAbsolutePath() + "\" isn't a file. Skipping");
        }

        try {
            var zipFile = new ZipFile(file);
            var entries = zipFile.entries();

            ZipEntry currentEntry;
            while (entries.hasMoreElements()) {
                currentEntry = entries.nextElement();

                if (!currentEntry.isDirectory() && currentEntry.getName().endsWith(".class")) {
                    try (var stream = zipFile.getInputStream(currentEntry)) {
                        var wrapper = ClassWrapper.fromLib(new ClassReader(stream));
                        classpath.put(wrapper.getName(), wrapper);
                    } catch (Throwable t) {
                        Logger.warn(String.format("Error while loading library class: \"%s\"", currentEntry.getName()));

                        if (PolymorphismData.VERBOSE) {
                            t.printStackTrace(System.err);
                        }
                    }
                }
            }
        } catch (ZipException e) {
            Logger.warn("Library \"" + file.getAbsolutePath() + "\" couldn't be loaded as a ZIP. Skipping");

            if (PolymorphismData.VERBOSE) {
                e.printStackTrace(System.err);
            }
        } catch (IOException e) {
            Logger.warn("Library \"" + file.getAbsolutePath() + "\" couldn't be loaded due to an IO error" + e.getMessage());

            if (PolymorphismData.VERBOSE) {
                e.printStackTrace(System.err);
            }
        }
    }

    public void loadAsInput(String path) {
        Logger.info("Loading input \"" + path + "\"");
        var file = new File(path);

        if (!file.exists()) {
            throw new PreventableObfuscationException("Input \"" + file.getAbsolutePath() + "\" doesn't exist");
        }
        if (!file.canRead()) {
            throw new PreventableObfuscationException("Input \"" + file.getAbsolutePath() + "\" doesn't have read permissions");
        }
        if (!file.isFile()) {
            throw new PreventableObfuscationException("Input \"" + file.getAbsolutePath() + "\" isn't a file");
        }

        try {
            var zipFile = new ZipFile(file);
            var entries = zipFile.entries();

            ZipEntry currentEntry;
            while (entries.hasMoreElements()) {
                currentEntry = entries.nextElement();

                if (!currentEntry.isDirectory()) {
                    try (var stream = zipFile.getInputStream(currentEntry)) {
                        if (currentEntry.getName().endsWith(".class")) {
                            try {
                                var wrapper = ClassWrapper.from(new ClassReader(stream));
                                classes.put(wrapper.getName(), wrapper);
                                classpath.put(wrapper.getName(), wrapper);
                            } catch (Throwable t) {
                                Logger.warn(String.format("Error while loading input class: \"%s\" (loading as resources instead)", currentEntry.getName()));
                                resources.put(currentEntry.getName(), IOUtils.toByteArray(stream));
                            }
                        } else {
                            resources.put(currentEntry.getName(), IOUtils.toByteArray(stream));
                        }
                    }
                }
            }
        } catch (ZipException e) {
            throw new PreventableObfuscationException("Input \"" + file.getAbsolutePath() + "\" couldn't be loaded as a ZIP. (" + e.getMessage() + ")");
        } catch (IOException e) {
            throw new PreventableObfuscationException("Input \"" + file.getAbsolutePath() + "\" couldn't be due to an IO error. (" + e.getMessage() + ")");
        }
    }

}
