package cc.polymorphism.obfuscator.analysis;

import cc.polymorphism.obfuscator.asm.wrapper.MethodWrapper;
import org.objectweb.asm.Type;
import org.objectweb.asm.tree.VarInsnNode;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

import static org.objectweb.asm.Opcodes.*;

public class LocalVariableAnalyser {
    public static void computeLocalVariables(MethodWrapper methodWrapper) {
        final var methodNode = methodWrapper.getMethodNode();

        final var argumentCount = Type.getArgumentTypes(methodNode.desc).length;
        final List<LocalVariableObject> objects = new ArrayList<>();

        for (final var vin : Stream.of(methodNode.instructions.toArray())
                .filter(ain -> ain instanceof VarInsnNode)
                .map(ain -> (VarInsnNode) ain)
                .toList()) {
            if (!Modifier.isStatic(methodNode.access)
                    && vin.var == 0) {
                continue;
            }

            if (objects.stream()
                    .anyMatch(lvo -> lvo.idx() == vin.var)) {
                continue;
            }

            if (vin.var <= argumentCount) {
                continue;
            }

            switch (vin.getOpcode()) {
                case ISTORE, ILOAD -> objects.add(new LocalVariableObject(vin.var, LocalVariableObjectType.INT));
                case LSTORE, LLOAD -> objects.add(new LocalVariableObject(vin.var, LocalVariableObjectType.LONG));
                case DSTORE, DLOAD -> objects.add(new LocalVariableObject(vin.var, LocalVariableObjectType.DOUBLE));
                case FSTORE, FLOAD -> objects.add(new LocalVariableObject(vin.var, LocalVariableObjectType.FLOAT));
                case ASTORE, ALOAD -> objects.add(new LocalVariableObject(vin.var, LocalVariableObjectType.OBJECT));
            }
        }

        methodWrapper.getLocals().clear();
        methodWrapper.getLocals().addAll(objects);
    }
}