package cc.polymorphism.obfuscator;

import cc.polymorphism.obfuscator.cli.CmdLineParser;
import cc.polymorphism.obfuscator.exceptions.obfuscation.FatalObfuscationException;
import cc.polymorphism.obfuscator.exceptions.obfuscation.PreventableObfuscationException;
import cc.polymorphism.obfuscator.logging.Logger;

import java.io.File;

public class PolymorphismEntrypoint {
    @SuppressWarnings("SameParameterValue")
    private static void bootstrap0(final boolean cliMode) {
        if (cliMode) {
            CmdLineParser.registerSwitch("help", 0);
            CmdLineParser.registerSwitch("config", 1);
            Logger.info("Finished setting up the command line.");
        }
    }

    public static void bootstrap(boolean cliMode) {
        bootstrap0(cliMode);
        Logger.info("Finished bootstrapping.");
    }

    private static int cli(String[] args) {
        var parser = new CmdLineParser();
        parser.parse(args);

        if (parser.containsSwitch("help")) {
            var programName = new File(PolymorphismEntrypoint.class.getProtectionDomain().getCodeSource().getLocation().getPath()).getName();

            Logger.info(String.format("Usage: %5s java -jar %s --config example.json", "", programName));
            Logger.info(String.format("Help Menu: %5s java -jar %s --help", "", programName));
        }

        if (parser.containsSwitch("config")) {
            var configPath = parser.getArgsFor("config")[0];
            var configFile = new File(configPath);

            try {
                var obfuscator = new Polymorphism(configFile);
                obfuscator.run();
            } catch (FatalObfuscationException e) {
                Logger.severe("A fatal exception was thrown: " + e.getMessage());

                if (PolymorphismData.VERBOSE) {
                    e.printStackTrace(System.out);
                }
                return -1;
            } catch (PreventableObfuscationException e) {
                Logger.severe("A preventable exception was thrown: " + e.getMessage());

                if (PolymorphismData.VERBOSE) {
                    e.printStackTrace(System.out);
                }
                return 1;
            } catch (Throwable t) {
                Logger.severe("An unknown error was throw: ");

                if (PolymorphismData.VERBOSE) {
                    t.printStackTrace(System.out);
                }
                return -1;
            }
        } else {
            Logger.info("Unknown operation: perhaps try viewing the results of --help?");
        }

        // Thank the Lord
        Logger.info("Exiting gracefully");
        return 0;
    }

    public static void main(String[] args) {
        bootstrap(true);
        System.exit(cli(args));
    }
}
