package cc.polymorphism.obfuscator;

import cc.polymorphism.obfuscator.asm.JarLoader;
import cc.polymorphism.obfuscator.asm.JarWriter;
import cc.polymorphism.obfuscator.asm.wrapper.ClassWrapper;
import cc.polymorphism.obfuscator.cli.config.ConfigHandler;
import cc.polymorphism.obfuscator.engine.hash.HashGenerator;
import cc.polymorphism.obfuscator.engine.seed.SeedGenerator;
import cc.polymorphism.obfuscator.engine.seed.v2.SeedGeneratorV2;
import cc.polymorphism.obfuscator.exceptions.missing.MissingClassException;
import cc.polymorphism.obfuscator.logging.Logger;
import cc.polymorphism.obfuscator.mutator.MutatorManager;
import lombok.Getter;
import lombok.Setter;

import java.io.File;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

@Getter
@Setter
public class Polymorphism {
    private static Polymorphism polymorphism;

    private final File configFile;

    private final HashGenerator hashGenerator;
    private final SeedGenerator seedGenerator;
    private final MutatorManager mutatorManager;

    private Map<String, ClassWrapper> classes;
    private Map<String, ClassWrapper> classpath;
    private Map<String, byte[]> resources;

    public Polymorphism(File configFile) {
        this.configFile = configFile;

        this.hashGenerator = new HashGenerator(this);
        this.seedGenerator = new SeedGenerator(this);
        this.mutatorManager = new MutatorManager(this);

        polymorphism = this;
    }

    public static synchronized Polymorphism getInstance() {
        return polymorphism;
    }

    public void run() {
        mutatorManager.init();

        try {
            if (!configFile.exists()) {
                if (configFile.createNewFile()) {
                    ConfigHandler.write(this, configFile);
                }

                Logger.info("Config didn't exist, a default one was generated for you!");
                return;
            }
        } catch (Exception exception) {
            exception.printStackTrace(System.err);
            return;
        }

        final var config = ConfigHandler.read(this, configFile);

        // ========================== Load input & libs
        var loader = new JarLoader();
        loader.loadAsInput(config.getInput());
        config.getDependencies().forEach(loader::loadAsLib);

        this.classes = loader.getClasses();
        this.classpath = loader.getClasspath();
        this.resources = loader.getResources();

        this.classes.values().forEach(classWrapper -> {
            classWrapper.getMethods().forEach(SeedGeneratorV2::render);
        });

        this.mutatorManager.transform();

        this.seedGenerator.render();
        this.hashGenerator.render();

        // ========================== Write output
        var writer = new JarWriter();
        writer.write(config.getOutput());
    }

    private void buildHierarchy(ClassWrapper wrapper, ClassWrapper sub, Set<String> visited) {
        if (visited.add(wrapper.getName())) {
            if (wrapper.getSuperName() != null) {
                var superParent = getClasspathWrapper(wrapper.getSuperName());
                wrapper.getParents().add(superParent);
                buildHierarchy(superParent, wrapper, visited);
            }
            if (wrapper.getInterfaceNames() != null) {
                wrapper.getInterfaceNames().forEach(interfaceName -> {
                    var interfaceParent = getClasspathWrapper(interfaceName);
                    wrapper.getParents().add(interfaceParent);
                    buildHierarchy(interfaceParent, wrapper, visited);
                });
            }
        }
        if (sub != null) {
            wrapper.getChildren().add(sub);
        }
    }

    public void buildHierarchyGraph() {
        HashSet<String> visited = new HashSet<>();
        classes.values().forEach(wrapper -> buildHierarchy(wrapper, null, visited));
    }

    public ClassWrapper getClasspathWrapper(String name) {
        var wrapper = classpath.get(name);

        if (wrapper == null) {
            throw MissingClassException.forLibraryClass(name);
        }

        return wrapper;
    }
}
