package cc.polymorphism.common.matcher;

import org.objectweb.asm.tree.AbstractInsnNode;

import java.util.ArrayList;
import java.util.List;

public class InstructionMatcher {
    private final InstructionPattern pattern;
    private final AbstractInsnNode start;
    private final List<List<AbstractInsnNode>> captured = new ArrayList<>();

    public InstructionMatcher(InstructionPattern pattern, AbstractInsnNode start) {
        this.pattern = pattern;
        this.start = start;
    }

    public boolean matches() {
        var rules = pattern.getRules();
        if (rules.size() == 0) {
            return false;
        }

        var current = start;
        ArrayList<AbstractInsnNode> possibleMatch = new ArrayList<>();
        for (var rule : rules) {
            if (current == null || !rule.matches(this, current)) {
                return false;
            }
            possibleMatch.add(current);
            current = current.getNext();
        }
        if (current != null) {
            return false;
        }
        captured.add(possibleMatch);
        return true;
    }

    public boolean find() { // This is a good example of why you shouldn't let me algorithmic programming
        var rules = pattern.getRules();
        if (rules.size() == 0) {
            return false;
        }
        var initialSize = captured.size();
        ArrayList<AbstractInsnNode> possibleMatch = new ArrayList<>();
        var current = start;
        int currentRuleIndex = 0;

        while (current != null) {
            if (rules.get(currentRuleIndex).matches(this, current)) {
                possibleMatch.add(current);
                if (++currentRuleIndex == rules.size()) {
                    captured.add(possibleMatch);
                    currentRuleIndex = 0;
                    possibleMatch = new ArrayList<>();
                }
            } else {
                currentRuleIndex = 0;
                possibleMatch.clear();
            }
            current = current.getNext();
        }

        return captured.size() - initialSize != 0;
    }

    public List<AbstractInsnNode> getCaptured(int which) {
        return captured.get(which);
    }

    public List<List<AbstractInsnNode>> getAllCaptured() {
        return captured;
    }
}
