package cc.polymorphism.common;

import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.tree.MethodNode;

/**
 * Determines the numbers of locals slots being used by a method based on the variable instructions within the method.
 *
 * @author itzsomebody
 */
public class MaxLocalsUpdater {
    private final MethodNode methodNode;

    private MaxLocalsUpdater(MethodNode methodNode) {
        this.methodNode = methodNode;
    }

    private int computeMaxs() {
        int size = 0;
        if ((methodNode.access & Opcodes.ACC_STATIC) == 0) {
            size += 1;
        }
        for (var type : Type.getArgumentTypes(methodNode.desc)) {
            size += type.getSize();
        }
        var visitor = new MaxLocalsVisitor(size);
        methodNode.accept(visitor);
        return visitor.getSize();
    }

    public static void update(MethodNode methodNode) {
        methodNode.visitMaxs(methodNode.maxStack, new MaxLocalsUpdater(methodNode).computeMaxs());
    }

    public static class MaxLocalsVisitor extends MethodVisitor {
        private int size;

        public MaxLocalsVisitor(int initialSize) {
            super(Opcodes.ASM9, null);
            this.size = initialSize;
        }

        @Override
        public void visitVarInsn(int opcode, int var) {
            if (var >= size) {
                size = var + 1;
                if (opcode == Opcodes.LLOAD
                        || opcode == Opcodes.DLOAD
                        || opcode == Opcodes.LSTORE
                        || opcode == Opcodes.DSTORE) {
                    size += 1;
                }
            }
        }

        @Override
        public void visitIincInsn(int var, int increment) {
            if (var >= size) {
                size = var + 1;
            }
        }

        public int getSize() {
            return size;
        }
    }
}
