package cc.polymorphism.assembly.expressions.predefined;

import cc.polymorphism.assembly.WrappedHandle;
import cc.polymorphism.assembly.WrappedType;
import org.junit.Assert;
import org.junit.Test;
import org.objectweb.asm.ConstantDynamic;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.tree.IntInsnNode;
import org.objectweb.asm.tree.LdcInsnNode;

import java.lang.invoke.MethodHandles;
import java.util.Collections;

import static cc.polymorphism.assembly.expressions.IRExpressions.*;

public class IRConstantTester {
    @Test
    public void testDynamicConst() throws Exception {
        var block = dynamicConstant("tux", WrappedType.from(String.class), WrappedHandle.getInvokeStaticHandle(IRConstantTester.class.getMethod("dynamicTester", MethodHandles.Lookup.class, String.class, Class.class)), Collections.emptyList()).bake();
        var insn = block.compile().getFirst();

        Assert.assertTrue(insn instanceof LdcInsnNode && ((LdcInsnNode) insn).cst instanceof ConstantDynamic);

        var dynamic = (ConstantDynamic) ((LdcInsnNode) insn).cst;
        var handle = dynamic.getBootstrapMethod();

        Assert.assertEquals(Type.getInternalName(IRConstantTester.class), handle.getOwner());
        Assert.assertEquals("dynamicTester", handle.getName());
        Assert.assertEquals("(Ljava/lang/invoke/MethodHandles$Lookup;Ljava/lang/String;Ljava/lang/Class;)Ljava/lang/String;", handle.getDesc());
        Assert.assertEquals("tux", dynamic.getName());
        Assert.assertEquals("Ljava/lang/String;", dynamic.getDescriptor());
    }

    @Test
    public void testNullConst() {
        var block = nullConstant(String.class).bake();
        var insns = block.compile();

        Assert.assertEquals(Opcodes.ACONST_NULL, insns.get(0).getOpcode());
    }

    @Test
    public void testIntConst() {
        Assert.assertEquals(Opcodes.ICONST_0, intConstant(0).bake().compile().get(0).getOpcode());
        Assert.assertEquals(Opcodes.BIPUSH, intConstant(Byte.MAX_VALUE).bake().compile().get(0).getOpcode());
        Assert.assertEquals(Byte.MAX_VALUE, ((IntInsnNode) intConstant(Byte.MAX_VALUE).bake().compile().get(0)).operand);
        Assert.assertEquals(Opcodes.SIPUSH, intConstant(Short.MAX_VALUE).bake().compile().get(0).getOpcode());
        Assert.assertEquals(Short.MAX_VALUE, ((IntInsnNode) intConstant(Short.MAX_VALUE).bake().compile().get(0)).operand);
        Assert.assertEquals(Opcodes.LDC, intConstant(Integer.MAX_VALUE).bake().compile().get(0).getOpcode());
        Assert.assertEquals(Integer.MAX_VALUE, ((LdcInsnNode) intConstant(Integer.MAX_VALUE).bake().compile().get(0)).cst);
    }

    @Test
    public void testLongConst() {
        Assert.assertEquals(Opcodes.LCONST_0, longConstant(0L).bake().compile().get(0).getOpcode());
        Assert.assertEquals(Opcodes.LDC, longConstant(Long.MAX_VALUE).bake().compile().get(0).getOpcode());
        Assert.assertEquals(Long.MAX_VALUE, ((LdcInsnNode) longConstant(Long.MAX_VALUE).bake().compile().get(0)).cst);
    }

    @Test
    public void testFloatConst() {
        Assert.assertEquals(Opcodes.FCONST_0, floatConstant(0F).bake().compile().get(0).getOpcode());
        Assert.assertEquals(Opcodes.LDC, floatConstant(Float.MAX_VALUE).bake().compile().get(0).getOpcode());
        Assert.assertEquals(Float.MAX_VALUE, ((LdcInsnNode) floatConstant(Float.MAX_VALUE).bake().compile().get(0)).cst);
    }

    @Test
    public void testDoubleConst() {
        Assert.assertEquals(Opcodes.DCONST_0, doubleConstant(0D).bake().compile().get(0).getOpcode());
        Assert.assertEquals(Opcodes.LDC, doubleConstant(Double.MAX_VALUE).bake().compile().get(0).getOpcode());
        Assert.assertEquals(Double.MAX_VALUE, ((LdcInsnNode) doubleConstant(Double.MAX_VALUE).bake().compile().get(0)).cst);
    }

    @Test
    public void testStringConst() {
        Assert.assertEquals(Opcodes.LDC, stringConstant("tux").bake().compile().get(0).getOpcode());
        Assert.assertEquals("tux", ((LdcInsnNode) stringConstant("tux").bake().compile().get(0)).cst);
    }

    @Test
    public void testClassConst() {
        Assert.assertEquals(Opcodes.LDC, classConstant(String.class).bake().compile().get(0).getOpcode());
        Assert.assertEquals(Type.getType(String.class), ((LdcInsnNode) classConstant(String.class).bake().compile().get(0)).cst);
    }

    public static String dynamicTester(MethodHandles.Lookup lookup, String name, Class<?> type) {
        return "test";
    }
}
