const express = require('express');
const bodyParser = require('body-parser');
const { MongoClient } = require('mongodb');
const crypto = require('crypto');
const fs = require('fs').promises;
const path = require('path');
const { ConsoleLogger, ConsoleConfig, Colors } = require('@fluxbot/better-console');
require('dotenv').config();
const cfonts = require('cfonts');
const { Sendhook } = require('./utils/webhook')
console.clear();
cfonts.say('Volt', {
  font: 'block',
  align: 'center',
  colors: ['system', 'blueBright'],
  background: 'transparent',
  letterSpacing: 1,
  lineHeight: 1,
  space: true,
  maxLength: '0',
  gradient: false,
  independentGradient: false,
  transitionGradient: false,
  reverse: false,
  env: 'node'
})


const log = new ConsoleLogger(new ConsoleConfig());
const app = express();
const PORT = process.env.PORT || 3344;
const routesPath = path.join(__dirname, 'routes');
const wsPath = path.join(__dirname, 'ws');

app.use(bodyParser.json());

const webHookURL = process.env.WEBHOOK_URL;
const mongoUri = process.env.MONGO_URI;
const dbName = process.env.DB_NAME;
const collectionName = process.env.COLLECTION_NAME;

let db;
let collection;
let loadedRoutes = 0;
let loadedWebSockets = 0;

async function connectToMongoDB() {
  try {
    const client = await MongoClient.connect(mongoUri, {
      serverSelectionTimeoutMS: 5000,
    });
    db = client.db(dbName);
    collection = db.collection(collectionName);
    log.info('Connected to MongoDB');
    return true;
  } catch (error) {
    console.error('Failed to connect to MongoDB:', error);
    return false;
  }
}

const encryptData = (data, key) => {
  const iv = crypto.randomBytes(16);
  const cipher = crypto.createCipheriv('aes-256-cbc', key, iv);
  let encrypted = cipher.update(data, 'utf8', 'hex');
  encrypted += cipher.final('hex');
  return { iv: iv.toString('hex'), encryptedData: encrypted };
};

async function loadRoutes() {
  try {
    if (!collection) {
      throw new Error('Cannot load routes: MongoDB collection not initialized');
    }

    const files = await fs.readdir(routesPath);
    for (const file of files) {
      if (file.endsWith('.js')) {
        try {
          const routeModule = require(path.join(routesPath, file));
          if (typeof routeModule === 'function') {
            routeModule(app, db, collection, encryptData, webHookURL);
            log.info(`Loaded route from: ${file}`);
            loadedRoutes++;
          } else {
            log.warn(`Skipping invalid route module: ${file}`);
          }
        } catch (routeError) {
          log.error(`Failed to load route ${file}:`, routeError);
        }
      }
    }
    log.info(`Loaded ${loadedRoutes} routes`);
  } catch (error) {
    console.error('Failed to load routes:', error);
    throw error;
  }
}

async function loadWebSockets() {
  try {
    const files = await fs.readdir(wsPath);
    for (const file of files) {
      if (file.endsWith('.js')) {
        try {
          const wsModule = require(path.join(wsPath, file));
          if (typeof wsModule === 'function') {
            wsModule(app, db, collection, encryptData, webHookURL);
            log.info(`Loaded websocket from: ${file}`);
            loadedWebSockets++;
          } else {
            log.warn(`Skipping invalid websocket module: ${file}`);
          }
        } catch (wsError) {
          log.error(`Failed to load websocket ${file}:`, wsError);
        }
      }
    }
    log.info(`Loaded ${loadedWebSockets} websockets`);
  } catch (error) {
    console.error('Failed to load websockets:', error);
    throw error;
  }
}

async function initializeServer() {
  try {
    const isConnected = await connectToMongoDB();
    if (!isConnected) {
      throw new Error('Failed to establish MongoDB connection');
    }

    if (!webHookURL || !mongoUri || !dbName || !collectionName) {
      throw new Error('Missing required environment variables');
    }

    await loadRoutes();
    await loadWebSockets();

    app.listen(PORT, () => {
      process.title = `[Volt Auth] | [Port] ${PORT} | [Routes] ${loadedRoutes} | [WebSockets] ${loadedWebSockets}`;
      log.custom('Server', Colors.Green, `Server is running on port ${PORT}`);
      const description = `\`\`\`ansi
[2;30m[[0m[2;36mBackend is now running[0m[2;30m][0m
\`\`\``
      Sendhook('Server', description, 0x00ff00)
    });
  } catch (error) {
    console.error('Failed to initialize server:', error);
    process.exit(1);
  }
}

initializeServer();